/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.weave.v2.module.DataFormat;
import org.mule.tooling.client.api.dataweave.DataWeaveModuleOption;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.dataweave.DataWeaveService;

import java.util.List;
import java.util.Map;

import static org.mule.weave.v2.module.pojo.JavaDataFormat.isJavaMimeType;

public class DefaultDataWeaveService implements DataWeaveService {

  private final DataWeaveRunnerProvider runnerProvider;
  private final ModulesAnalyzer modulesAnalyzer;

  public DefaultDataWeaveService(DataWeaveRunnerProvider runnerProvider, ModulesAnalyzer modulesAnalyzer) {
    this.runnerProvider = runnerProvider;
    this.modulesAnalyzer = modulesAnalyzer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    final EventModel event = parameters.getEvent();
    if (event != null && anyInputIsJavaBased(event)) {
      throw new IllegalArgumentException("Java input not supported, serialize to DW script");
    }
    return getRunnerProvider().getRunner(parameters).execute(parameters);
  }

  private boolean anyInputIsJavaBased(final EventModel event) {
    final MessageModel message = event.getMessage();
    if (isJavaBased(message.getPayload()) || isJavaBased(message.getAttributes())) {
      return true;
    }

    final Map<String, TypedValueModel> variables = event.getVariables();
    if (variables != null && variables.values().stream().anyMatch(this::isJavaBased)) {
      return true;
    }

    // otherwise
    return false;
  }

  private boolean isJavaBased(final TypedValueModel model) {
    return model != null && isJavaMimeType(model.getDataType().getMediaType());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<DataWeaveModuleOption> getReaderOptions(String contentType) {
    return getModulesAnalyzer().getOptions(contentType, DataFormat::readerOptions);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<DataWeaveModuleOption> getWriterOptions(String contentType) {
    return getModulesAnalyzer().getOptions(contentType, DataFormat::writerOptions);
  }

  public ModulesAnalyzer getModulesAnalyzer() {
    return modulesAnalyzer;
  }

  public DataWeaveRunnerProvider getRunnerProvider() {
    return runnerProvider;
  }
}
