/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.core.api.util.IOUtils.closeQuietly;
import static org.mule.runtime.deployment.model.api.application.ApplicationDescriptor.DEFAULT_CONFIGURATION_RESOURCE_LOCATION;
import static org.mule.runtime.deployment.model.api.application.ApplicationDescriptor.MULE_ARTIFACT_JSON_DESCRIPTOR;
import static org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor.MULE_ARTIFACT_FOLDER;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.DefaultApplicationModel;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.config.spring.dsl.processor.ConfigFile;
import org.mule.runtime.core.component.config.ResourceProvider;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;

import org.apache.commons.io.IOUtils;
import org.apache.commons.io.input.BoundedInputStream;

/**
 * Implementation that handles a {@link URL} to a remote zip file with the application content compressed.
 *
 * @since 4.0
 */
public class ApplicationModelFactoryFromUrl extends BaseApplicationModelFactory {

  private static final String MULE_APPLICATION_JSON_PATH = MULE_ARTIFACT_FOLDER + "/" + MULE_ARTIFACT_JSON_DESCRIPTOR;
  private static final String APP_TYPES_PATH = MULE + "/" + APP_TYPES_DATA;

  private static final int CONNECT_TIMEOUT = 5000;
  private static final int READ_TIMEOUT = 5000;

  /**
   * {@inheritDoc}
   */
  public ApplicationModelFactoryFromUrl(ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    super(componentBuildingDefinitionLoader);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ApplicationModel> createApplicationModel(URL applicationUrl, Set<ExtensionModel> extensionModels) {
    final byte[] buffer = readToBuffer(applicationUrl);
    return processZip(buffer, extensionModels)
        .map(applicationModel -> new DefaultApplicationModel(applicationUrl.getPath(), applicationModel, readTypesData(buffer)));
  }

  private String readTypesData(byte[] buffer) {
    try (ZipInputStream zipInputStream = new ZipInputStream(new ByteArrayInputStream(buffer))) {
      ZipEntry entry;
      while ((entry = zipInputStream.getNextEntry()) != null) {
        if (entry.getName().equals(APP_TYPES_PATH)) {
          return org.apache.commons.io.IOUtils.toString(zipInputStream);
        }
      }
      return null;
    } catch (IOException e) {
      throw new ToolingException(e);
    }
  }

  private byte[] readToBuffer(URL url) {
    InputStream inputStream = null;
    try {
      URLConnection urlConnection = url.openConnection();
      urlConnection.setConnectTimeout(CONNECT_TIMEOUT);
      urlConnection.setReadTimeout(READ_TIMEOUT);
      inputStream = new BufferedInputStream(urlConnection.getInputStream());
      ByteArrayOutputStream bufferOutputStream = new ByteArrayOutputStream();
      IOUtils.copy(inputStream, bufferOutputStream);
      return bufferOutputStream.toByteArray();
    } catch (IOException e) {
      throw new ToolingException(e);
    } finally {
      closeQuietly(inputStream);
    }
  }

  private Optional<Stream<String>> findConfigsFromDeployProperties(byte[] buffer) {
    Optional<Stream<String>> configsOptional = empty();
    ZipEntry entry;
    try (ZipInputStream zipInputStream = new ZipInputStream(new ByteArrayInputStream(buffer))) {
      while ((entry = zipInputStream.getNextEntry()) != null) {
        if (entry.getName().equals(MULE_APPLICATION_JSON_PATH)) {
          final MuleApplicationModel muleApplicationModel =
              new MuleApplicationModelJsonSerializer().deserialize(org.apache.commons.io.IOUtils.toString(zipInputStream));
          if (!muleApplicationModel.getConfigs().isEmpty()) {
            return of(muleApplicationModel.getConfigs().stream()
                .map(configFileName -> Paths.get(MULE, configFileName).toString()));
          }
        }
      }
    } catch (IOException e) {
      throw new ToolingException(e);
    }
    return of(configsOptional.orElse(Stream.of(DEFAULT_CONFIGURATION_RESOURCE_LOCATION)));
  }

  private Optional<org.mule.runtime.config.spring.dsl.model.ApplicationModel> readConfigs(Stream<String> configs, byte[] buffer,
                                                                                          Set<ExtensionModel> extensionModels)
      throws IOException {
    List<String> configsList = configs.collect(toList());
    try (ZipInputStream zipInputStream = new ZipInputStream(new ByteArrayInputStream(buffer))) {
      ZipEntry entry;
      List<ConfigFile> configFiles = new ArrayList();
      while ((entry = zipInputStream.getNextEntry()) != null) {
        final String path = new File(entry.getName()).getPath();
        if (configsList.contains(path)) {
          final BoundedInputStream boundedInputStream = new BoundedInputStream(zipInputStream);
          boundedInputStream.setPropagateClose(false);
          loadConfigFile(path, boundedInputStream, extensionModels).map(configLine -> configFiles.add(configLine));
        }
      }
      try {
        ResourceProvider externalResourceProvider = new ZipResourceProvider(buffer);
        return of(loadApplicationModel(configFiles, extensionModels, externalResourceProvider));
      } catch (Exception e) {
        throw new ToolingException(e);
      }
    }
  }

  private Optional<org.mule.runtime.config.spring.dsl.model.ApplicationModel> processZip(byte[] buffer,
                                                                                         Set<ExtensionModel> extensionModels) {
    return findConfigsFromDeployProperties(buffer).map(configs -> {
      try {
        return readConfigs(configs, buffer, extensionModels);
      } catch (IOException e) {
        throw new ToolingException(e);
      }
    }).orElse(empty());
  }

  private class ZipResourceProvider implements ResourceProvider {

    byte[] buffer;

    public ZipResourceProvider(byte[] buffer) {
      this.buffer = buffer;
    }

    @Override
    public InputStream getResourceAsStream(String uri) {
      ZipEntry entry;
      ZipInputStream zipInputStream = null;
      List<String> searchPaths = getSearchPaths();
      try {
        zipInputStream = new ZipInputStream(new ByteArrayInputStream(buffer));
        while ((entry = zipInputStream.getNextEntry()) != null) {
          final ZipEntry finalEntry = entry;
          if (searchPaths.stream().anyMatch(path -> finalEntry.getName().equals(path + "/" + uri))) {
            return zipInputStream;
          }
        }
      } catch (IOException e) {
        IOUtils.closeQuietly(zipInputStream);
        throw new ToolingException(e);
      }
      IOUtils.closeQuietly(zipInputStream);
      return null;
    }
  }
}
