/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.collect.Lists.newArrayList;
import static com.google.common.io.Files.createTempDir;
import static java.util.Arrays.asList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.core.api.util.Pair;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginClassLoaderFactory;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.DefaultRegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.internal.plugin.BundlePluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingPluginClassLoaderBuilder;
import org.mule.runtime.module.artifact.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.classloader.DeployableArtifactClassLoaderFactory;
import org.mule.runtime.module.artifact.classloader.MuleDeployableArtifactClassLoader;
import org.mule.runtime.module.deployment.impl.internal.artifact.DescriptorLoaderRepository;
import org.mule.runtime.module.deployment.impl.internal.artifact.ExtensionModelDiscoverer;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorLoader;
import org.mule.runtime.module.deployment.impl.internal.plugin.MuleExtensionModelLoaderManager;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginMavenClassLoaderModelLoader;
import org.mule.runtime.module.extension.internal.loader.ExtensionModelLoaderRepository;
import org.mule.runtime.module.extension.internal.resources.MuleExtensionModelProvider;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.extension.ExtensionModelService;

import com.mulesoft.mule.runtime.config.spring.dsl.MuleEeExtensionModelProvider;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Default implementation of {@link ExtensionModelService}.
 *
 * @since 4.0
 */
public class DefaultExtensionModelService implements InternalExtensionModelService, Initializable {

  private MavenClientHolder mavenClientHolder;
  private MavenClient mavenClient;
  private ModuleRepository moduleRepository = null;

  private ArtifactPluginDescriptorFactory artifactPluginDescriptorFactory;
  private ArtifactPluginDescriptorLoader artifactPluginDescriptorLoader;
  private ArtifactClassLoader containerArtifactClassLoader;
  private ExtensionModelDiscoverer extensionModelDiscoverer = new ExtensionModelDiscoverer();

  /**
   * Creates an instance of the service.
   */
  public DefaultExtensionModelService(MavenClientHolder mavenClientHolder, ModuleRepository moduleRepository,
                                      ArtifactClassLoader containerArtifactClassLoader) {
    this.mavenClientHolder = mavenClientHolder;
    this.moduleRepository = moduleRepository;
    this.containerArtifactClassLoader = containerArtifactClassLoader;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void init() {
    mavenClient = mavenClientHolder.get();
    MavenClientProvider mavenClientProvider = discoverProvider(this.getClass().getClassLoader());

    // Maven Loaders should use current MavenClient configuration
    DescriptorLoaderRepository descriptorLoaderRepository =
        new ToolingDescriptorLoaderRepository(new PluginMavenClassLoaderModelLoader(mavenClient, mavenClientProvider
            .getLocalRepositorySuppliers()));
    artifactPluginDescriptorFactory = new ArtifactPluginDescriptorFactory(descriptorLoaderRepository);
    artifactPluginDescriptorLoader = new ArtifactPluginDescriptorLoader(artifactPluginDescriptorFactory);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<ExtensionModel> loadMuleExtensionModels() {
    return asList(MuleExtensionModelProvider.getExtensionModel(), MuleEeExtensionModelProvider.getExtensionModel());
  }

  private Optional<Pair<ExtensionModel, ArtifactClassLoader>> resolveExtensionModel(ArtifactPluginDescriptor artifactPluginDescriptor)

      throws IOException, MuleException {
    ToolingPluginClassLoaderBuilder builder =
        new ToolingPluginClassLoaderBuilder(new TemporaryArtifactClassLoaderFactory(),
                                            new BundlePluginDependenciesResolver(artifactPluginDescriptorFactory),
                                            artifactPluginDescriptor,
                                            new DefaultRegionPluginClassLoadersFactory(new ArtifactPluginClassLoaderFactory(),
                                                                                       moduleRepository));
    builder.setParentClassLoader(containerArtifactClassLoader);
    ArtifactClassLoader pluginClassLoader = builder.build();

    MuleExtensionModelLoaderManager extensionModelLoaderRepository =
        new MuleExtensionModelLoaderManager(containerArtifactClassLoader);
    extensionModelLoaderRepository.start();
    Optional<ExtensionModel> extensionModel =
        getExtensionModel(artifactPluginDescriptor, pluginClassLoader, extensionModelLoaderRepository);
    if (!extensionModel.isPresent()) {
      pluginClassLoader.dispose();
      return empty();
    }
    return of(new Pair<>(extensionModel.get(), pluginClassLoader));
  }

  private Optional<ExtensionModel> getExtensionModel(ArtifactPluginDescriptor artifactPluginDescriptor,
                                                     ArtifactClassLoader pluginClassLoader,
                                                     ExtensionModelLoaderRepository extensionModelLoaderRepository) {
    Set<ExtensionModel> extensionModels =
        extensionModelDiscoverer.discoverExtensionModels(extensionModelLoaderRepository,
                                                         newArrayList(new Pair(artifactPluginDescriptor, pluginClassLoader)));
    if (extensionModels.isEmpty()) {
      return empty();
    }
    return of(extensionModels.iterator().next());
  }

  @Override
  public Optional<Pair<ExtensionModel, ArtifactClassLoader>> loadPairArtifactClassLoaderAndExtensionModel(ArtifactDescriptor pluginDescriptor) {
    BundleDescriptor.Builder builder = new BundleDescriptor.Builder();
    builder.setGroupId(pluginDescriptor.getGroupId())
        .setArtifactId(pluginDescriptor.getArtifactId())
        .setType(pluginDescriptor.getExtension())
        .setVersion(pluginDescriptor.getVersion())
        .setClassifier(pluginDescriptor.getClassifier())
        .build();
    return loadPairArtifactClassLoaderAndExtensionModel(new File(mavenClient.resolveBundleDescriptor(builder.build())
        .getBundleUri()));
  }

  @Override
  public Optional<Pair<ExtensionModel, ArtifactClassLoader>> loadPairArtifactClassLoaderAndExtensionModel(File file) {
    File tempFolder = createTempDir();
    try {
      ArtifactPluginDescriptor artifactPluginDescriptor;
      artifactPluginDescriptor = artifactPluginDescriptorLoader.load(file);
      return resolveExtensionModel(artifactPluginDescriptor);
    } catch (Exception e) {
      throw new ToolingException("Error while loading ExtensionModel for plugin: " + file.getAbsolutePath(), e);
    } finally {
      deleteQuietly(tempFolder);
    }
  }

  /**
   * Creates a class loader instance for a temporary artifact.
   */
  private class TemporaryArtifactClassLoaderFactory
      implements DeployableArtifactClassLoaderFactory<org.mule.runtime.module.artifact.descriptor.ArtifactDescriptor> {

    /**
     * {@inheritDoc}
     */
    @Override
    public ArtifactClassLoader create(String artifactId, ArtifactClassLoader parent,
                                      org.mule.runtime.module.artifact.descriptor.ArtifactDescriptor descriptor,
                                      List<ArtifactClassLoader> artifactPluginClassLoaders) {
      return new MuleDeployableArtifactClassLoader(artifactId, descriptor, descriptor.getClassLoaderModel().getUrls(),
                                                   parent.getClassLoader(),
                                                   parent.getClassLoaderLookupPolicy(), artifactPluginClassLoaders);
    }

  }

}
