/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.collect.Maps.newHashMap;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;
import static org.mule.tooling.client.internal.CollectionUtils.copy;
import static org.mule.tooling.client.internal.ExtensionModelPartsFactory.*;

import org.mule.tooling.client.api.extension.model.Category;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.ImportedTypeModel;
import org.mule.tooling.client.api.extension.model.MuleVersion;
import org.mule.tooling.client.api.extension.model.SubTypesModel;
import org.mule.tooling.client.api.extension.model.XmlDslModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.operation.RouterModel;
import org.mule.tooling.client.api.extension.model.operation.ScopeModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;

import java.util.List;
import java.util.Map;

/**
 * Factory for {@link org.mule.tooling.client.api.extension.model.ExtensionModel}. Responsible for adapting a Mule Runtime
 * {@link org.mule.runtime.api.meta.model.ExtensionModel} and implements flyweight pattern in order to reduce the amount of objects created.
 *
 * @since 4.0
 */
public class ExtensionModelFactory {

  // Caches for implementing flyweight pattern
  private Map<String, ConnectionProviderModel> connectionProviderModelsCache = newHashMap();
  private Map<String, SourceModel> sourceModelsCache = newHashMap();
  private Map<String, OperationModel> operationModelsCache = newHashMap();
  private Map<String, ScopeModel> scopeModelsCache = newHashMap();
  private Map<String, RouterModel> routerModelsCache = newHashMap();
  private Map<String, FunctionModel> functionModelsCache = newHashMap();


  public ExtensionModel createExtensionModel(org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel) {
    return new ExtensionModel(
                              runtimeExtensionModel.getName(),
                              toCategoryDTO(runtimeExtensionModel.getCategory()),
                              new MuleVersion(runtimeExtensionModel.getMinMuleVersion().toCompleteNumericVersion()),
                              runtimeExtensionModel.getVendor(),
                              copy(runtimeExtensionModel.getResources()),
                              runtimeExtensionModel.getImportedTypes().stream()
                                  .map(importedTypeModel -> new ImportedTypeModel(importedTypeModel.getImportedType())).collect(
                                                                                                                                toSet()),
                              runtimeExtensionModel
                                  .getSubTypes().stream().map(subTypesModel -> new SubTypesModel(subTypesModel.getBaseType(),
                                                                                                 subTypesModel.getSubTypes()))
                                  .collect(
                                           toSet()),
                              runtimeExtensionModel.getTypes(),
                              runtimeExtensionModel.getVersion(),
                              new XmlDslModel(runtimeExtensionModel.getXmlDslModel().getXsdFileName(),
                                              runtimeExtensionModel.getXmlDslModel().getSchemaVersion(),
                                              runtimeExtensionModel.getXmlDslModel().getPrefix(),
                                              runtimeExtensionModel.getXmlDslModel().getNamespace(),
                                              runtimeExtensionModel.getXmlDslModel().getSchemaLocation()),
                              runtimeExtensionModel.getDescription(),
                              toDisplayModelTO(runtimeExtensionModel.getDisplayModel()),
                              toExternalLibraryModelsDTO(runtimeExtensionModel.getExternalLibraryModels()),
                              toConfigurationModelsDTO(runtimeExtensionModel),
                              toOperationModelsDTO(runtimeExtensionModel.getOperationModels()),
                              toConnectionProvidersDTO(runtimeExtensionModel.getConnectionProviders()),
                              toSourceModelsDTO(runtimeExtensionModel.getSourceModels()),
                              toFunctionModelsDTO(runtimeExtensionModel.getFunctionModels()));
  }

  private static Category toCategoryDTO(org.mule.runtime.api.meta.Category category) {
    if (category == org.mule.runtime.api.meta.Category.COMMUNITY) {
      return Category.communityCategory(category.name());
    }
    if (category == org.mule.runtime.api.meta.Category.SELECT) {
      return Category.selectCategory(category.name());
    }
    if (category == org.mule.runtime.api.meta.Category.PREMIUM) {
      return Category.premiumCategory(category.name());
    }
    if (category == org.mule.runtime.api.meta.Category.CERTIFIED) {
      return Category.certifiedCategory(category.name());
    }
    return new Category(category.name());
  }

  private List<ConfigurationModel> toConfigurationModelsDTO(org.mule.runtime.api.meta.model.ExtensionModel serializedExtensionModel) {
    return serializedExtensionModel.getConfigurationModels().stream()
        .map(configurationModel -> new ConfigurationModel(configurationModel.getName(),
                                                          configurationModel.getDescription(),
                                                          toParameterGroupModelsDTO(configurationModel
                                                              .getParameterGroupModels()),
                                                          toOperationModelsDTO(configurationModel.getOperationModels()),
                                                          toExternalLibraryModelsDTO(configurationModel
                                                              .getExternalLibraryModels()),
                                                          toSourceModelsDTO(configurationModel.getSourceModels()),
                                                          toConnectionProvidersDTO(configurationModel.getConnectionProviders()),
                                                          toDisplayModelTO(configurationModel.getDisplayModel())))
        .collect(toList());
  }

  private List<OperationModel> toOperationModelsDTO(List<org.mule.runtime.api.meta.model.operation.OperationModel> operationModels) {
    return operationModels.stream()
        .map(operationModel -> toOperationModelDTO(operationModel, new CachedComponentModelVisitorDTO()))
        .collect(toList());
  }

  private List<SourceModel> toSourceModelsDTO(List<org.mule.runtime.api.meta.model.source.SourceModel> sourceModels) {
    return sourceModels.stream()
        .map(sourceModel -> sourceModel == null ? null
            : sourceModelsCache.computeIfAbsent(sourceModel.getName(), name -> toSourceModelDTO(sourceModel)))
        .collect(toList());
  }

  private List<ConnectionProviderModel> toConnectionProvidersDTO(List<org.mule.runtime.api.meta.model.connection.ConnectionProviderModel> connectionProviders) {
    return connectionProviders.stream()
        .map(connectionProviderModel -> connectionProviderModelsCache.computeIfAbsent(connectionProviderModel.getName(),
                                                                                      name -> toConnectionProviderDTO(connectionProviderModel)))
        .collect(toList());
  }

  private List<FunctionModel> toFunctionModelsDTO(List<org.mule.runtime.api.meta.model.function.FunctionModel> functionModels) {
    return functionModels.stream()
        .map(functionModel -> functionModelsCache.computeIfAbsent(functionModel.getName(),
                                                                  name -> toFunctionModelDTO(functionModel)))
        .collect(toList());
  }

  private class CachedComponentModelVisitorDTO extends ComponentModelVisitorDTO {

    @Override
    public void visit(org.mule.runtime.api.meta.model.operation.OperationModel operationModel) {
      operationModelDTO = operationModelsCache.computeIfAbsent(operationModel.getName(),
                                                               name -> {
                                                                 super.visit(operationModel);
                                                                 return operationModelDTO;
                                                               });
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.operation.ScopeModel scopeModel) {
      operationModelDTO = scopeModelsCache.computeIfAbsent(scopeModel.getName(),
                                                           name -> {
                                                             super.visit(scopeModel);
                                                             return (ScopeModel) operationModelDTO;
                                                           });
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.operation.RouterModel routerModel) {
      operationModelDTO = routerModelsCache.computeIfAbsent(routerModel.getName(),
                                                            name -> {
                                                              super.visit(routerModel);
                                                              return (RouterModel) operationModelDTO;
                                                            });
    }
  }

}
