/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptyList;
import static org.mule.tooling.client.internal.CollectionUtils.copy;
import static org.mule.runtime.api.meta.model.parameter.ElementReference.ElementType.CONFIG;
import static org.mule.runtime.api.meta.model.parameter.ElementReference.ElementType.FLOW;

import org.mule.runtime.api.meta.model.util.ComponentModelVisitor;
import org.mule.runtime.extension.internal.property.MetadataKeyPartModelProperty;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.location.Location;
import org.mule.tooling.client.api.extension.model.DisplayModel;
import org.mule.tooling.client.api.extension.model.parameter.ElementReference;
import org.mule.tooling.client.api.extension.model.ExternalLibraryModel;
import org.mule.tooling.client.api.extension.model.ExternalLibraryType;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExpressionSupport;
import org.mule.tooling.client.api.extension.model.LayoutModel;
import org.mule.tooling.client.api.extension.model.OutputModel;
import org.mule.tooling.client.api.extension.model.ParameterDslConfiguration;
import org.mule.tooling.client.api.extension.model.PathModel;
import org.mule.tooling.client.api.extension.model.PathModelType;
import org.mule.tooling.client.api.extension.model.Stereotype;
import org.mule.tooling.client.api.extension.model.connection.ConnectionManagementType;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.metadata.MetadataKeyPartModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.operation.RouteModel;
import org.mule.tooling.client.api.extension.model.operation.RouterModel;
import org.mule.tooling.client.api.extension.model.operation.ScopeModel;
import org.mule.tooling.client.api.extension.model.parameter.ExclusiveParametersModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterRole;
import org.mule.tooling.client.api.extension.model.source.SourceCallbackModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.api.extension.model.value.ValueProviderModel;

import java.util.LinkedList;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import static java.util.Collections.emptySet;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toSet;

/**
 * Utils for creating the Extension Model DTOs from the Mule-API objects
 *
 * @since 4.0
 */
public class ExtensionModelPartsFactory {

  public static Set<ExternalLibraryModel> toExternalLibraryModelsDTO(
                                                                     Set<org.mule.runtime.api.meta.model.ExternalLibraryModel> externalLibraryModels) {
    return externalLibraryModels.stream().map(externalLibraryModel -> new ExternalLibraryModel(externalLibraryModel.getName(),
                                                                                               externalLibraryModel
                                                                                                   .getDescription(),
                                                                                               externalLibraryModel
                                                                                                   .getRequiredClassName()
                                                                                                   .orElse(null),
                                                                                               externalLibraryModel
                                                                                                   .getRegexMatcher()
                                                                                                   .orElse(null),
                                                                                               toExternalLibraryTypeDTO(externalLibraryModel
                                                                                                   .getType())))
        .collect(toSet());
  }

  private static ExternalLibraryType toExternalLibraryTypeDTO(org.mule.runtime.api.meta.ExternalLibraryType type) {
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.NATIVE) {
      return ExternalLibraryType.nativeExternalLibraryType(type.name());
    }
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.JAR) {
      return ExternalLibraryType.jarExternalLibraryType(type.name());
    }
    if (type == org.mule.runtime.api.meta.ExternalLibraryType.DEPENDENCY) {
      return ExternalLibraryType.dependencyExternalLibraryType(type.name());
    }
    return new ExternalLibraryType(type.name());
  }


  public static ConnectionProviderModel toConnectionProviderDTO(org.mule.runtime.api.meta.model.connection.ConnectionProviderModel connectionProviderModel) {
    return new ConnectionProviderModel(connectionProviderModel.getName(),
                                       connectionProviderModel.getDescription(),
                                       toDisplayModelTO(connectionProviderModel.getDisplayModel()),
                                       toConnectionManagementTypeDTO(connectionProviderModel.getConnectionManagementType()),
                                       toParameterGroupModelsDTO(connectionProviderModel.getParameterGroupModels()),
                                       toExternalLibraryModelsDTO(connectionProviderModel.getExternalLibraryModels()));
  }

  private static ConnectionManagementType toConnectionManagementTypeDTO(
                                                                        org.mule.runtime.api.meta.model.connection.ConnectionManagementType connectionManagementType) {
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.POOLING) {
      return ConnectionManagementType.poolingConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.CACHED) {
      return ConnectionManagementType.cachedConnectionManagementType(connectionManagementType.name());
    }
    if (connectionManagementType == org.mule.runtime.api.meta.model.connection.ConnectionManagementType.NONE) {
      return ConnectionManagementType.noneConnectionManagementType(connectionManagementType.name());
    }

    return new ConnectionManagementType(connectionManagementType.name());
  }


  public static SourceModel toSourceModelDTO(org.mule.runtime.api.meta.model.source.SourceModel sourceModel) {
    if (sourceModel == null) {
      return null;
    }
    return new SourceModel(sourceModel.getName(),
                           sourceModel.getDescription(),
                           toParameterGroupModelsDTO(sourceModel.getParameterGroupModels()),
                           toDisplayModelTO(sourceModel.getDisplayModel()),
                           toOutputModelDTO(sourceModel.getOutput()),
                           toOutputModelDTO(sourceModel.getOutputAttributes()),
                           sourceModel.isTransactional(),
                           sourceModel.requiresConnection(),
                           sourceModel.supportsStreaming(),
                           toStereotypesDTO(sourceModel.getStereotypes()),
                           sourceModel.hasResponse(),
                           toSourceCallbackModelDTO(sourceModel.getSuccessCallback()),
                           toSourceCallbackModelDTO(sourceModel.getErrorCallback()),
                           toSourceCallbackModelDTO(sourceModel.getTerminateCallback()),
                           toErrorModelsDTO(sourceModel.getErrorModels()));
  }

  private static Set<ErrorModel> toErrorModelsDTO(Set<org.mule.runtime.api.meta.model.error.ErrorModel> errorModels) {
    return errorModels.stream()
        .map(errorModel -> toErrorModelDTO(errorModel))
        .collect(toSet());
  }

  private static ErrorModel toErrorModelDTO(org.mule.runtime.api.meta.model.error.ErrorModel errorModel) {
    if (errorModel == null) {
      return null;
    }
    return new ErrorModel(errorModel.getType(), errorModel.getNamespace(),
                          toErrorModelDTO(errorModel.getParent().orElse(null)));
  }

  private static SourceCallbackModel toSourceCallbackModelDTO(Optional<org.mule.runtime.api.meta.model.source.SourceCallbackModel> sourceCallbackModelOptional) {
    return sourceCallbackModelOptional
        .map(sourceCallbackModel -> new SourceCallbackModel(sourceCallbackModel.getName(),
                                                            sourceCallbackModel.getDescription(),
                                                            toParameterGroupModelsDTO(sourceCallbackModel
                                                                .getParameterGroupModels()),
                                                            toDisplayModelTO(sourceCallbackModel.getDisplayModel())))
        .orElse(null);
  }

  private static Set<Stereotype> toStereotypesDTO(Set<org.mule.runtime.api.meta.model.Stereotype> stereotypes) {
    // Mule Extension Models (json written by hand) has null value for this property
    if (stereotypes == null) {
      return emptySet();
    }
    return stereotypes.stream()
        .map(stereotype -> new Stereotype(stereotype.getName()))
        .collect(toSet());
  }

  private static OutputModel toOutputModelDTO(org.mule.runtime.api.meta.model.OutputModel output) {
    return new OutputModel(output.getDescription(), output.getType(), output.hasDynamicType());
  }

  public static OperationModel toOperationModelDTO(org.mule.runtime.api.meta.model.operation.OperationModel operationModel) {
    if (operationModel == null) {
      return null;
    }
    ComponentModelVisitorDTO visitor = new ComponentModelVisitorDTO();
    operationModel.accept(visitor);
    return visitor.operationModelDTO;
  }

  public static OperationModel toOperationModelDTO(org.mule.runtime.api.meta.model.operation.OperationModel operationModel,
                                                   ComponentModelVisitorDTO visitor) {
    if (operationModel == null) {
      return null;
    }
    operationModel.accept(visitor);
    return visitor.operationModelDTO;
  }

  private static List<RouteModel> toRouteModelsDTO(List<org.mule.runtime.api.meta.model.operation.RouteModel> routeModels) {
    return routeModels.stream()
        .map(routeModel -> new RouteModel(routeModel.getName(),
                                          routeModel.getDescription(),
                                          routeModel.getMinOccurs(),
                                          routeModel.getMaxOccurs().orElse(null),
                                          toStereotypesDTO(routeModel.getAllowedStereotypes().orElse(emptySet())),
                                          toParameterGroupModelsDTO(routeModel.getParameterGroupModels())))
        .collect(toList());
  }

  public static List<ParameterGroupModel> toParameterGroupModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterGroupModel> parameterGroupModels) {
    return parameterGroupModels.stream()
        .map(parameterGroupModel -> new ParameterGroupModel(parameterGroupModel.getName(),
                                                            parameterGroupModel.getDescription(),
                                                            toDisplayModelTO(parameterGroupModel.getDisplayModel()),
                                                            toLayoutModelDTO(parameterGroupModel.getLayoutModel()),
                                                            toParameterModelsDTO(parameterGroupModel.getParameterModels()),
                                                            toExclusiveParameterModelsDTO(parameterGroupModel
                                                                .getExclusiveParametersModels()),
                                                            parameterGroupModel.isShowInDsl()))
        .collect(toList());
  }

  private static List<ExclusiveParametersModel> toExclusiveParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ExclusiveParametersModel> exclusiveParametersModels) {
    return exclusiveParametersModels.stream()
        .map(exclusiveParametersModel -> new ExclusiveParametersModel(copy(exclusiveParametersModel
            .getExclusiveParameterNames()),
                                                                      exclusiveParametersModel.isOneRequired()))
        .collect(toList());
  }

  private static List<ParameterModel> toParameterModelsDTO(List<org.mule.runtime.api.meta.model.parameter.ParameterModel> parameterModels) {
    return parameterModels.stream()
        .map(parameterModel -> new ParameterModel(parameterModel.getName(),
                                                  parameterModel.getDescription(),
                                                  toDisplayModelTO(parameterModel.getDisplayModel()),
                                                  parameterModel.getType(),
                                                  parameterModel.hasDynamicType(),
                                                  parameterModel.isRequired(),
                                                  parameterModel.isOverrideFromConfig(),
                                                  toExpressionSupportDTO(parameterModel.getExpressionSupport()),
                                                  parameterModel.getDefaultValue(),
                                                  toParameterRoleDTO(parameterModel.getRole()),
                                                  toParameterDslConfigurationDTO(parameterModel.getDslConfiguration()),
                                                  toLayoutModelDTO(parameterModel.getLayoutModel()),
                                                  toMetadataKeyPartModel(parameterModel
                                                      .getModelProperty(MetadataKeyPartModelProperty.class)),
                                                  toValueProviderModelDTO(parameterModel.getValueProviderModel()),
                                                  toElementReferencesDTO(parameterModel.getElementReferences())))
        .collect(toList());
  }

  private static List<ElementReference> toElementReferencesDTO(List<org.mule.runtime.api.meta.model.parameter.ElementReference> refs) {
    if (refs == null) {
      return emptyList();
    }
    return refs.stream()
        .map(ExtensionModelPartsFactory::toElementReferenceDTO)
        .collect(toList());
  }

  private static ElementReference toElementReferenceDTO(org.mule.runtime.api.meta.model.parameter.ElementReference er) {
    if (er.getType().equals(CONFIG)) {
      return ElementReference.configElementReference(er.getType().toString(), er.getNamespace(), er.getElementName());
    }
    if (er.getType().equals(FLOW)) {
      return ElementReference.flowElementReference(er.getType().toString(), er.getNamespace(), er.getElementName());
    }
    return new ElementReference(er.getType().toString(), er.getNamespace(), er.getElementName());
  }

  private static MetadataKeyPartModel toMetadataKeyPartModel(Optional<MetadataKeyPartModelProperty> modelProperty) {
    return modelProperty.map(metadataKeyPartModelProperty -> new MetadataKeyPartModel(metadataKeyPartModelProperty.getOrder()))
        .orElse(null);
  }

  private static ExpressionSupport toExpressionSupportDTO(org.mule.runtime.api.meta.ExpressionSupport expressionSupport) {
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.SUPPORTED) {
      return ExpressionSupport.supportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.NOT_SUPPORTED) {
      return ExpressionSupport.notSupportedExpressionSupport(expressionSupport.name());
    }
    if (expressionSupport == org.mule.runtime.api.meta.ExpressionSupport.REQUIRED) {
      return ExpressionSupport.requiredExpressionSupport(expressionSupport.name());
    }
    return new ExpressionSupport(expressionSupport.name());
  }

  private static ParameterDslConfiguration toParameterDslConfigurationDTO(
                                                                          org.mule.runtime.api.meta.model.ParameterDslConfiguration dslConfiguration) {
    return new ParameterDslConfiguration(dslConfiguration.allowsInlineDefinition(), dslConfiguration.allowTopLevelDefinition(),
                                         dslConfiguration.allowsReferences());
  }

  private static LayoutModel toLayoutModelDTO(Optional<org.mule.runtime.api.meta.model.display.LayoutModel> layoutModelOptional) {
    return layoutModelOptional.map(layoutModel -> new LayoutModel(layoutModel.getTabName().orElse(null),
                                                                  layoutModel.getOrder().orElse(null),
                                                                  layoutModel.isPassword(),
                                                                  layoutModel.isText(),
                                                                  layoutModel.isQuery()))
        .orElse(null);
  }

  private static ParameterRole toParameterRoleDTO(org.mule.runtime.api.meta.model.parameter.ParameterRole parameterRole) {
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.BEHAVIOUR) {
      return ParameterRole.behaviourParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.CONTENT) {
      return ParameterRole.contentParameterRole(parameterRole.name());
    }
    if (parameterRole == org.mule.runtime.api.meta.model.parameter.ParameterRole.PRIMARY_CONTENT) {
      return ParameterRole.primaryContentParameterRole(parameterRole.name());
    }
    return new ParameterRole(parameterRole.name());
  }

  public static DisplayModel toDisplayModelTO(Optional<org.mule.runtime.api.meta.model.display.DisplayModel> displayModelOptional) {
    return displayModelOptional.map(displayModel -> new DisplayModel(displayModel.getDisplayName(), displayModel.getSummary(),
                                                                     displayModel.getExample(),
                                                                     toPathModelDTO(displayModel.getPathModel())))
        .orElse(null);
  }

  private static PathModel toPathModelDTO(Optional<org.mule.runtime.api.meta.model.display.PathModel> pathModelOptional) {
    return pathModelOptional.map(pathModel -> new PathModel(toPathModelTypeDTO(pathModel.getType()), pathModel.acceptsUrls(),
                                                            pathModel.getFileExtensions().toArray(new String[0])))
        .orElse(null);
  }

  private static PathModelType toPathModelTypeDTO(org.mule.runtime.api.meta.model.display.PathModel.Type type) {
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.DIRECTORY) {
      return PathModelType.directoryPathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.FILE) {
      return PathModelType.fileyPathModelType(type.name());
    }
    if (type == org.mule.runtime.api.meta.model.display.PathModel.Type.ANY) {
      return PathModelType.anyPathModelType(type.name());
    }
    return new PathModelType(type.name());
  }

  private static ValueProviderModel toValueProviderModelDTO(Optional<org.mule.runtime.api.meta.model.parameter.ValueProviderModel> valueProviderModel) {
    return valueProviderModel
        .map(model -> new ValueProviderModel(model.getActingParameters(), model.getPartOrder(), model.getProviderName()))
        .orElse(null);
  }

  public static Location toLocationDTO(org.mule.runtime.api.component.location.Location location) {
    if (location == null) {
      return null;
    }
    String global = location.getGlobalElementName();
    List<String> rest = location.getParts();

    LinkedList<String> parts = new LinkedList<>();
    parts.add(global);
    parts.addAll(rest);

    return new Location(parts);
  }

  public static FunctionModel toFunctionModelDTO(org.mule.runtime.api.meta.model.function.FunctionModel functionModel) {
    return new FunctionModel(functionModel.getName(), functionModel.getDescription(),
                             toParameterGroupModelsDTO(functionModel.getParameterGroupModels()),
                             toDisplayModelTO(functionModel.getDisplayModel()), toOutputModelDTO(functionModel.getOutput()));
  }

  public static class ComponentModelVisitorDTO implements ComponentModelVisitor {

    protected OperationModel operationModelDTO;

    @Override
    public void visit(org.mule.runtime.api.meta.model.operation.OperationModel operationModel) {
      operationModelDTO = new OperationModel(operationModel.getName(),
                                             operationModel.getDescription(),
                                             toParameterGroupModelsDTO(operationModel.getParameterGroupModels()),
                                             toDisplayModelTO(operationModel.getDisplayModel()),
                                             operationModel.isBlocking(),
                                             toErrorModelsDTO(operationModel.getErrorModels()),
                                             toOutputModelDTO(operationModel.getOutput()),
                                             toOutputModelDTO(operationModel.getOutputAttributes()),
                                             operationModel.isTransactional(),
                                             operationModel.requiresConnection(),
                                             operationModel.supportsStreaming(),
                                             toStereotypesDTO(operationModel.getStereotypes()));
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.operation.ScopeModel scopeModel) {
      operationModelDTO = new ScopeModel(scopeModel.getName(),
                                         scopeModel.getDescription(),
                                         toParameterGroupModelsDTO(scopeModel.getParameterGroupModels()),
                                         toDisplayModelTO(scopeModel.getDisplayModel()),
                                         scopeModel.isBlocking(),
                                         toErrorModelsDTO(scopeModel.getErrorModels()),
                                         toOutputModelDTO(scopeModel.getOutput()),
                                         toOutputModelDTO(scopeModel.getOutputAttributes()),
                                         scopeModel.isTransactional(),
                                         scopeModel.requiresConnection(),
                                         scopeModel.supportsStreaming(),
                                         toStereotypesDTO(scopeModel.getStereotypes()));
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.operation.RouterModel routerModel) {
      operationModelDTO = new RouterModel(routerModel.getName(),
                                          routerModel.getDescription(),
                                          toParameterGroupModelsDTO(routerModel.getParameterGroupModels()),
                                          toDisplayModelTO(routerModel.getDisplayModel()),
                                          routerModel.isBlocking(),
                                          toErrorModelsDTO(routerModel.getErrorModels()),
                                          toOutputModelDTO(routerModel.getOutput()),
                                          toOutputModelDTO(routerModel.getOutputAttributes()),
                                          routerModel.isTransactional(),
                                          routerModel.requiresConnection(),
                                          routerModel.supportsStreaming(),
                                          toStereotypesDTO(routerModel.getStereotypes()),
                                          toRouteModelsDTO(routerModel.getRouteModels()));
    }

    @Override
    public void visit(org.mule.runtime.api.meta.model.source.SourceModel sourceModel) {
      // Nothing to do here...
      operationModelDTO = null;
    }
  }

}
