/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static org.mule.runtime.config.spring.api.XmlConfigurationDocumentLoader.noValidationDocumentLoader;
import static java.util.Optional.ofNullable;
import static org.mule.runtime.config.spring.internal.util.ComponentBuildingDefinitionUtils.registerComponentBuildingDefinitions;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.runtime.core.api.util.collection.Collectors.toImmutableList;

import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.config.spring.api.dsl.model.ApplicationModel;
import org.mule.runtime.config.spring.api.dsl.model.ComponentBuildingDefinitionRegistry;
import org.mule.runtime.config.spring.api.dsl.model.ResourceProvider;
import org.mule.runtime.config.spring.api.dsl.processor.ArtifactConfig;
import org.mule.runtime.config.spring.api.dsl.processor.ConfigFile;
import org.mule.runtime.config.spring.api.dsl.processor.ConfigLine;
import org.mule.runtime.config.spring.api.dsl.processor.xml.XmlApplicationParser;
import org.mule.runtime.config.spring.api.dsl.xml.StaticXmlNamespaceInfo;
import org.mule.runtime.config.spring.api.dsl.xml.StaticXmlNamespaceInfoProvider;
import org.mule.runtime.core.api.registry.ServiceRegistry;
import org.mule.runtime.core.api.registry.SpiServiceRegistry;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfo;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;

import com.google.common.collect.ImmutableList;

import java.io.File;
import java.io.InputStream;
import java.net.URI;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.w3c.dom.Document;

/**
 * Base implementation for {@link ApplicationModelFactory}.
 *
 * @since 4.0
 */
public abstract class BaseApplicationModelFactory implements ApplicationModelFactory {

  protected static final String APP_TYPES_DATA = "application-types.xml";
  protected static final String MULE = "mule";
  private ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader;

  /**
   * Creates a new instance.
   *
   * @param componentBuildingDefinitionLoader the loader to use for getting the
   *        {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinition}s from a
   *        {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinitionProvider}.
   */
  public BaseApplicationModelFactory(ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    this.componentBuildingDefinitionLoader = componentBuildingDefinitionLoader;
  }

  /**
   * Loads {@link ConfigFile} from the configuration file {@link InputStream}.
   *
   * @param fileName the name of the config file.
   * @param inputStream {@link InputStream} content of the file.
   * @param extensionModels {@link List} of {@link ExtensionModel} discovered from the application.
   * @return a optional {@link ConfigFile}.
   */
  protected Optional<ConfigFile> loadConfigFile(String fileName, InputStream inputStream, Set<ExtensionModel> extensionModels) {
    return withContextClassLoader(this.getClass().getClassLoader(), () -> {
      List<XmlNamespaceInfoProvider> xmlNamespaceInfoProviders =
          ImmutableList.<XmlNamespaceInfoProvider>builder()
              .add(createStaticNamespaceInfoProviders(extensionModels))
              .addAll(discoverRuntimeXmlNamespaceInfoProvider())
              .build();
      XmlApplicationParser xmlApplicationParser = new XmlApplicationParser(xmlNamespaceInfoProviders);
      Document document = noValidationDocumentLoader().loadDocument(fileName, inputStream);
      return xmlApplicationParser.parse(document.getDocumentElement())
          .map(configLine -> new ConfigFile(relativizeConfigFileName(fileName), singletonList(configLine)));
    });
  }

  private List<XmlNamespaceInfoProvider> discoverRuntimeXmlNamespaceInfoProvider() {
    ImmutableList.Builder namespaceInfoProvidersBuilder = ImmutableList.builder();
    namespaceInfoProvidersBuilder
        .addAll(new SpiServiceRegistry().lookupProviders(XmlNamespaceInfoProvider.class, this.getClass().getClassLoader()));
    return namespaceInfoProvidersBuilder.build();
  }

  private XmlNamespaceInfoProvider createStaticNamespaceInfoProviders(Set<ExtensionModel> extensionModels) {
    List<XmlNamespaceInfo> extensionNamespaces = extensionModels.stream()
        .map(ext -> new StaticXmlNamespaceInfo(ext.getXmlDslModel().getNamespace(), ext.getXmlDslModel().getPrefix()))
        .collect(toImmutableList());

    return new StaticXmlNamespaceInfoProvider(extensionNamespaces);
  }

  private static String getMuleAppsFolder() {
    return MULE;
  }

  private String relativizeConfigFileName(String fileName) {
    if (fileName == null) {
      return null;
    }

    final URI muleAppsURI = new File(getMuleAppsFolder()).toURI();
    final URI configFileURI = new File(fileName).toURI();
    return muleAppsURI.relativize(configFileURI).toString();
  }

  /**
   * Loads {@link ApplicationModel} from {@link ConfigLine}.
   *
   * @param configFiles {@link List configFiles} to be added to the {@link ApplicationModel}.
   * @param extensionModels {@link Set} of {@link ExtensionModel extension models} to be used for generating the application
   *        model.
   * @param externalResourceProvider the provider for configuration properties files and ${file::name.txt} placeholders
   * @return a {@link ApplicationModel}.
   */
  protected ApplicationModel loadApplicationModel(List<ConfigFile> configFiles, Set<ExtensionModel> extensionModels,
                                                  ResourceProvider externalResourceProvider)
      throws Exception {
    final ArtifactConfig.Builder builder = new ArtifactConfig.Builder();
    if (!configFiles.isEmpty()) {
      configFiles.forEach(configFile -> builder.addConfigFile(configFile));
    }
    ArtifactConfig artifactConfig = builder.build();
    ApplicationModel applicationModel =
        new ApplicationModel(artifactConfig, null, extensionModels, emptyMap(), of(new EmptyConfigurationProperties()),
                             of(createComponentBuildingDefinitionRegistry(extensionModels,
                                                                          componentBuildingDefinitionLoader)),
                             false, externalResourceProvider);
    return applicationModel;
  }

  private static ComponentBuildingDefinitionRegistry createComponentBuildingDefinitionRegistry(
                                                                                               Set<ExtensionModel> extensionModels,
                                                                                               ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    ServiceRegistry serviceRegistry = new SpiServiceRegistry();
    final ComponentBuildingDefinitionRegistry componentBuildingDefinitionRegistry =
        new ComponentBuildingDefinitionRegistry();

    registerComponentBuildingDefinitions(serviceRegistry, BaseApplicationModelFactory.class.getClassLoader(),
                                         componentBuildingDefinitionRegistry, ofNullable(extensionModels),
                                         componentBuildingDefinitionLoader.loadComponentBuildingDefinitions(extensionModels));

    return componentBuildingDefinitionRegistry;
  }

  protected List<String> getSearchPaths() {
    // TODO: update when MMP-184 is done
    return asList("classes");
  }

  /**
   * Implementation that allows to load an {@link ApplicationModel} without having the properties replaced.
   */
  private class EmptyConfigurationProperties implements ConfigurationProperties {

    @Override
    public <T> Optional<T> resolveProperty(String s) {
      return of((T) "");
    }

    @Override
    public Optional<Boolean> resolveBooleanProperty(String s) {
      return of(false);
    }

    @Override
    public Optional<String> resolveStringProperty(String s) {
      return of("");
    }

  }
}
