/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Throwables.propagateIfPossible;
import static org.apache.commons.io.FileUtils.toFile;
import static org.mule.tooling.client.internal.DataSensePartsFactory.toDataSenseComponentInfoDTO;
import static org.mule.tooling.client.internal.DataSensePartsFactory.toDataSenseInfoDTO;
import static org.mule.tooling.client.internal.TimeoutMethodUtils.withTimeout;

import org.mule.datasense.api.DataSense;
import org.mule.datasense.api.DataSenseResolutionScope;
import org.mule.datasense.api.DataSenseInfo;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.CachedDataSenseProvider;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataCacheProvider;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.api.metadataprovider.DefaultDataSenseProvider;
import org.mule.datasense.api.metadataprovider.ExtensionsProvider;
import org.mule.datasense.impl.DefaultDataSense;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.datasense.DataSenseResolveRequest;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.application.Application;

import com.google.common.collect.ImmutableSet;

import java.net.URL;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link DataSenseService}.
 *
 * @since 4.0
 */
public class DefaultDataSenseService implements DataSenseService {

  private static final String FILE_PROTOCOL = "file";

  private final Logger logger = LoggerFactory.getLogger(this.getClass());
  private final ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader;

  private DataSense dataSense = new DefaultDataSense();

  private Supplier<URL> applicationUrlSupplier;
  private MetadataProvider metadataProvider;

  private final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
  private final Lock readLock = readWriteLock.readLock();
  private final Lock writeLock = readWriteLock.writeLock();

  private ApplicationModelFactory applicationModelFactory;
  private ApplicationModel applicationModel;
  private ExtensionsProvider extensionsProviderSupplier;

  /**
   * Creates an instance of the DataSense service.
   *
   * @param applicationUrlSupplier {@link URL} {@link Supplier} with the location of the application content, either a file path
   *        with expanded content or a zip. Not null.
   * @param metadataProvider {@link MetadataProvider} to resolve dynamic Metadata if needed. Not null.
   * @param application {@link Application} over the one datasense is going to be resolved.
   * @param componentBuildingDefinitionLoader loader for {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinition}s
   */
  public DefaultDataSenseService(Supplier<URL> applicationUrlSupplier, MetadataProvider metadataProvider,
                                 Application application, ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    checkNotNull(applicationUrlSupplier, "applicationUrlSupplier cannot be null");
    checkNotNull(metadataProvider, "metadataProvider cannot be null");

    this.applicationUrlSupplier = applicationUrlSupplier;
    this.metadataProvider = metadataProvider;
    this.componentBuildingDefinitionLoader = componentBuildingDefinitionLoader;
    this.extensionsProviderSupplier = new ExtensionsProvider() {

      private Set<ExtensionModel> extensionModels;

      @Override
      public Set<ExtensionModel> getExtensions() {
        if (extensionModels == null) {
          extensionModels = ImmutableSet.<ExtensionModel>builder().addAll(application.getPluginDependencies()).build();
        }
        return extensionModels;
      }
    };
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<org.mule.tooling.client.api.datasense.DataSenseInfo> resolveDataSense(DataSenseRequest dataSenseRequest) {
    buildApplicationModel();

    Location location = dataSenseRequest.getLocation();
    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = dataSenseRequest.getDataSenseMetadataCacheProvider();

    if (logger.isDebugEnabled()) {
      logger.debug("Resolving DataSense for component location: {} on application: {}", dataSenseRequest.getLocation(),
                   applicationUrlSupplier.get());
    }

    return withTimeout(dataSenseRequest.getRequestTimeout(), () -> {
      try {
        ApplicationModel applicationModel = getApplicationModel(extensionsProviderSupplier.getExtensions());

        DataSenseProvider dataSenseProvider =
            getDataSenseProvider(extensionsProviderSupplier, dataSenseMetadataCacheProvider, dataSenseRequest);

        DataSenseResolutionScope dataSenseResolutionScope = dataSenseRequest instanceof DataSenseResolveRequest
            ? ((DataSenseResolveRequest) dataSenseRequest).getDataSenseResolutionScope() : null;
        if (dataSenseResolutionScope == null) {
          Optional<DataSenseInfo> dataSenseInfo = this.dataSense.resolve(location, applicationModel, dataSenseProvider);
          return toDataSenseInfoDTO(dataSenseInfo);
        } else {
          Optional<DataSenseInfo> dataSenseInfo =
              this.dataSense.resolve(dataSenseResolutionScope, applicationModel, dataSenseProvider);
          return toDataSenseInfoDTO(dataSenseInfo);
        }
      } catch (Exception e) {
        propagateIfPossible(e, MissingToolingConfigurationException.class);
        throw new ToolingException("Error while resolving DataSense for location: " + location + " on applicationUrl: "
            + applicationUrlSupplier.get(), e);
      }
    }, null);
  }

  private void buildApplicationModel() {
    // TODO (gfernandes) MULE-11149
    if (isApplicationExploded()) {
      if (logger.isDebugEnabled()) {
        logger.debug("Working with application expanded file path");
      }
      this.applicationModelFactory = new ApplicationModelFactoryFromExpandedFolder(componentBuildingDefinitionLoader);
    } else {
      if (logger.isDebugEnabled()) {
        logger.debug("Working with remote compressed application content");
      }
      this.applicationModelFactory = new ApplicationModelFactoryFromUrl(componentBuildingDefinitionLoader);
    }

  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<DataSenseComponentInfo> resolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    buildApplicationModel();

    Location location = dataSenseRequest.getLocation();
    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = dataSenseRequest.getDataSenseMetadataCacheProvider();

    checkNotNull(location, "location cannot be null");

    if (logger.isDebugEnabled()) {
      logger.debug("Resolving DataSense for location: {} on application: {}", dataSenseRequest.getLocation(),
                   applicationUrlSupplier.get());
    }

    return withTimeout(dataSenseRequest.getRequestTimeout(), () -> {
      try {
        ApplicationModel applicationModel = getApplicationModel(extensionsProviderSupplier.getExtensions());
        DataSenseProvider dataSenseProvider =
            getDataSenseProvider(extensionsProviderSupplier, dataSenseMetadataCacheProvider, dataSenseRequest);

        return toDataSenseComponentInfoDTO(this.dataSense.resolveComponent(location, applicationModel, dataSenseProvider));
      } catch (Exception e) {
        propagateIfPossible(e, MissingToolingConfigurationException.class);
        throw new ToolingException("Error while resolving DataSense for location: " + location + " on applicationUrl: "
            + applicationUrlSupplier.get(), e);
      }
    }, null);
  }

  private DataSenseProvider getDataSenseProvider(ExtensionsProvider extensionsProvider,
                                                 DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider,
                                                 DataSenseRequest request) {
    DataSenseProvider dataSenseProvider =
        new DefaultDataSenseProvider(extensionsProvider, new ToolingDataSenseMetadataProvider(metadataProvider,
                                                                                              request.getMetadataTimeout()));
    if (dataSenseMetadataCacheProvider != null) {
      dataSenseProvider = new CachedDataSenseProvider(applicationModel, dataSenseMetadataCacheProvider, dataSenseProvider);
    }
    return dataSenseProvider;
  }


  private boolean isApplicationExploded() {
    return applicationUrlSupplier.get().getProtocol().equals(FILE_PROTOCOL) && toFile(applicationUrlSupplier.get()).isDirectory();
  }

  private ApplicationModel getApplicationModel(Set<ExtensionModel> extensionModels) {
    readLock.lock();
    try {
      if (applicationModel != null) {
        return applicationModel;
      }
    } finally {
      readLock.unlock();
    }

    writeLock.lock();
    try {
      // check another thread didn't beat us to it
      if (applicationModel != null) {
        return applicationModel;
      }

      applicationModel = applicationModelFactory.createApplicationModel(applicationUrlSupplier.get(), extensionModels)
          .orElseThrow(() -> new ToolingException("Couldn't create ApplicationModel for application: "
              + applicationUrlSupplier.get()));
      return applicationModel;
    } finally {
      writeLock.unlock();
    }
  }


  @Override
  public Optional<DataSenseInfo> internalResolveDataSense(DataSenseRequest dataSenseRequest) {
    buildApplicationModel();

    Location location = dataSenseRequest.getLocation();
    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = dataSenseRequest.getDataSenseMetadataCacheProvider();

    if (logger.isDebugEnabled()) {
      logger.debug("Resolving DataSense for component location: {} on application: {}", dataSenseRequest.getLocation(),
                   applicationUrlSupplier.get());
    }

    return withTimeout(dataSenseRequest.getRequestTimeout(), () -> {
      try {
        ApplicationModel applicationModel = getApplicationModel(extensionsProviderSupplier.getExtensions());

        DataSenseProvider dataSenseProvider =
            getDataSenseProvider(extensionsProviderSupplier, dataSenseMetadataCacheProvider, dataSenseRequest);

        DataSenseResolutionScope dataSenseResolutionScope = dataSenseRequest instanceof DataSenseResolveRequest
            ? ((DataSenseResolveRequest) dataSenseRequest).getDataSenseResolutionScope() : null;
        if (dataSenseResolutionScope == null) {
          return this.dataSense.resolve(location, applicationModel, dataSenseProvider);
        } else {
          return this.dataSense.resolve(dataSenseResolutionScope, applicationModel, dataSenseProvider);
        }
      } catch (Exception e) {
        propagateIfPossible(e, MissingToolingConfigurationException.class);
        throw new ToolingException("Error while resolving DataSense for location: " + location + " on applicationUrl: "
            + applicationUrlSupplier.get(), e);
      }
    }, null);
  }

  @Override
  public Optional<org.mule.datasense.api.DataSenseComponentInfo> internalResolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    buildApplicationModel();

    Location location = dataSenseRequest.getLocation();
    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider = dataSenseRequest.getDataSenseMetadataCacheProvider();

    checkNotNull(location, "location cannot be null");

    if (logger.isDebugEnabled()) {
      logger.debug("Resolving DataSense for location: {} on application: {}", dataSenseRequest.getLocation(),
                   applicationUrlSupplier.get());
    }

    return withTimeout(dataSenseRequest.getRequestTimeout(), () -> {
      try {
        ApplicationModel applicationModel = getApplicationModel(extensionsProviderSupplier.getExtensions());
        DataSenseProvider dataSenseProvider =
            getDataSenseProvider(extensionsProviderSupplier, dataSenseMetadataCacheProvider, dataSenseRequest);

        return this.dataSense.resolveComponent(location, applicationModel, dataSenseProvider);
      } catch (Exception e) {
        propagateIfPossible(e, MissingToolingConfigurationException.class);
        throw new ToolingException("Error while resolving DataSense for location: " + location + " on applicationUrl: "
            + applicationUrlSupplier.get(), e);
      }
    }, null);
  }


}
