/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.beust.jcommander.internal.Lists.newArrayList;
import static com.google.common.collect.Maps.newHashMap;
import static java.lang.String.format;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.core.api.registry.SpiServiceRegistry;
import org.mule.runtime.module.artifact.descriptor.BundleDescriptorLoader;
import org.mule.runtime.module.artifact.descriptor.ClassLoaderModelLoader;
import org.mule.runtime.module.artifact.descriptor.DescriptorLoader;
import org.mule.runtime.module.deployment.impl.internal.artifact.DescriptorLoaderRepository;
import org.mule.runtime.module.deployment.impl.internal.artifact.LoaderNotFoundException;
import org.mule.runtime.module.deployment.impl.internal.artifact.ServiceRegistryDescriptorLoaderRepository;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginMavenClassLoaderModelLoader;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

/**
 * Implementation of {@link DescriptorLoaderRepository} similar to {@link ServiceRegistryDescriptorLoaderRepository} for Tooling
 * that allows to use a provided {@link MavenClient}.
 *
 * @since 4.0
 */
public class ToolingDescriptorLoaderRepository implements DescriptorLoaderRepository {

  private Map<Class, List<DescriptorLoader>> descriptorLoaders = newHashMap();

  public ToolingDescriptorLoaderRepository(PluginMavenClassLoaderModelLoader pluginMavenClassLoaderModelLoader) {
    descriptorLoaders.put(BundleDescriptorLoader.class,
                          findBundleDescriptorLoaders(BundleDescriptorLoader.class, new SpiServiceRegistry()));
    descriptorLoaders.put(ClassLoaderModelLoader.class, newArrayList(pluginMavenClassLoaderModelLoader));
  }

  private List<DescriptorLoader> findBundleDescriptorLoaders(Class<? extends DescriptorLoader> descriptorLoaderClass,
                                                             SpiServiceRegistry serviceRegistry) {
    List<DescriptorLoader> descriptorLoaders = new ArrayList<>();
    Collection<? extends DescriptorLoader> providers =
        serviceRegistry.lookupProviders(descriptorLoaderClass, this.getClass().getClassLoader());

    for (DescriptorLoader loader : providers) {
      descriptorLoaders.add(loader);
    }

    return descriptorLoaders;
  }

  @Override
  public <T extends DescriptorLoader> T get(String id, ArtifactType artifactType, Class<T> loaderClass)
      throws LoaderNotFoundException {
    DescriptorLoader descriptorLoader = null;
    List<DescriptorLoader> registeredDescriptorLoaders = descriptorLoaders.get(loaderClass);
    if (registeredDescriptorLoaders != null) {
      for (DescriptorLoader loader : registeredDescriptorLoaders) {
        if (loader.getId().equals(id) && loader.supportsArtifactType(artifactType)) {
          descriptorLoader = loader;
        }
      }
    }

    if (descriptorLoader == null) {
      throw new LoaderNotFoundException(format("There is no loader with ID='%s' and type '%s'", id, loaderClass.getName()));
    }

    return (T) descriptorLoader;
  }
}
