/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataKeysContainerDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataResultDTO;

import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.metadata.MetadataService;

import org.mule.tooling.client.internal.application.Deployable;

import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Public implementation of {@link MetadataService} that adapts Mule-API objects to DTOs
 *
 * @since 4.0
 */
public class ToolingMetadataServiceAdapter implements MetadataService {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private MetadataProvider metadataProvider;

  /**
   * Creates an instance of this service adapter.
   *
   * @param metadataProvider {@link Supplier} for a {@link Deployable}. Non null. Should return always the same instance.
   */
  public ToolingMetadataServiceAdapter(MetadataProvider metadataProvider) {
    checkNotNull(metadataProvider, "metadataProvider cannot be null");

    this.metadataProvider = metadataProvider;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public org.mule.tooling.client.api.metadata.MetadataResult<org.mule.tooling.client.api.metadata.MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException, ToolingException {

    MetadataResult<MetadataKeysContainer> metadataKeys = metadataProvider.getMetadataKeys(metadataKeysRequest);
    return toMetadataResultDTO(metadataKeys, () -> toMetadataKeysContainerDTO(metadataKeys.get()));
  }

  @Override
  public MetadataResult<MetadataKeysContainer> internalMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException, ToolingException {
    return metadataProvider.getMetadataKeys(metadataKeysRequest);
  }

}
