/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Optional.ofNullable;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static java.util.Optional.empty;

import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClient.Builder;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.internal.application.ApplicationService;
import org.mule.tooling.client.internal.application.DefaultApplicationService;

import java.io.InputStream;
import java.util.Optional;

/**
 * Default implementation of {@link Builder} that creates a {@link ToolingRuntimeClient}.
 *
 * @since 4.0
 */
public class DefaultToolingRuntimeClientBuilder implements Builder {

  private MavenClient mavenClient;
  private ModuleRepository moduleRepository;
  private ArtifactClassLoader containerClassLoaderFactory;
  private ExtensionModelServiceCache extensionModelServiceCache;
  private ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader;
  private ApplicationCache applicationCache;
  private Optional<InputStream> logConfig = empty();

  private AgentConfiguration agentConfiguration;

  public DefaultToolingRuntimeClientBuilder(MavenClient bootstrapMavenClient,
                                            ModuleRepository moduleRepository,
                                            ArtifactClassLoader containerClassLoaderFactory,
                                            ExtensionModelServiceCache extensionModelServiceCache,
                                            ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader,
                                            ApplicationCache applicationCache) {
    this.mavenClient = bootstrapMavenClient;
    this.moduleRepository = moduleRepository;
    this.containerClassLoaderFactory = containerClassLoaderFactory;
    this.extensionModelServiceCache = extensionModelServiceCache;
    this.componentBuildingDefinitionLoader = componentBuildingDefinitionLoader;
    this.applicationCache = applicationCache;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration) {
    this.agentConfiguration = agentConfiguration;
    return this;
  }

  @Override
  public Builder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
    this.mavenClient = discoverProvider(DefaultToolingRuntimeClientBuilder.class.getClassLoader())
        .createMavenClient(mavenConfiguration);
    return this;
  }

  @Override
  public Builder withLoggingConfiguration(InputStream input) {
    this.logConfig = ofNullable(input);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingRuntimeClient build() {
    MuleArtifactResourcesRegistry muleArtifactResourcesRegistry =
        new MuleArtifactResourcesRegistry(mavenClient, moduleRepository,
                                          containerClassLoaderFactory);
    InternalExtensionModelService extensionModelService =
        new DefaultExtensionModelService(mavenClient, muleArtifactResourcesRegistry);
    MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider =
        new CachedExtensionModelService(extensionModelServiceCache,
                                        extensionModelService);
    ApplicationService applicationService = new DefaultApplicationService(muleArtifactResourcesRegistry);

    return new DefaultToolingRuntimeClient(mavenClient,
                                           ofNullable(agentConfiguration), muleRuntimeExtensionModelProvider,
                                           componentBuildingDefinitionLoader, applicationService, applicationCache,
                                           logConfig);
  }

}
