/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.runtime.api.value.ResolvingFailure.Builder.newFailure;
import static org.mule.runtime.api.value.ValueResult.resultFrom;
import static org.mule.tooling.client.internal.TimeoutMethodUtils.withTimeout;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.api.value.ValueResult;
import org.mule.tooling.client.api.value.provider.ValueProviderRequest;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.RemoteApplicationInvoker;

/**
 * Provides the capability of resolve the {@link Value values} for any Value Provider capable element in a Mule App.
 *
 * @since 1.0
 */
public class DefaultValueProviderService implements ValueProviderService {

  private RemoteApplicationInvoker remoteApplicationInvoker;

  /**
   * Creates an instance of the service.
   *
   * @param remoteApplicationInvoker {@link Application} to deploy and resolve operations against a Mule Runtime. Non null.
   */
  public DefaultValueProviderService(RemoteApplicationInvoker remoteApplicationInvoker) {
    checkNotNull(remoteApplicationInvoker, "remoteApplicationInvoker cannot be null");

    this.remoteApplicationInvoker = remoteApplicationInvoker;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueResult getValues(ValueProviderRequest valueProviderRequest) {
    return withTimeout(valueProviderRequest.getRequestTimeout(),
                       () -> remoteApplicationInvoker.evaluateWithRemoteApplication((applicationId,
                                                                                     runtimeToolingService) -> runtimeToolingService
                                                                                         .getValues(applicationId,
                                                                                                    valueProviderRequest
                                                                                                        .getLocation(),
                                                                                                    valueProviderRequest
                                                                                                        .getProviderName())),
                       of(throwable -> resultFrom(newFailure(throwable).build())),
                       empty());
  }
}
