/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Preconditions;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;

import com.google.common.base.Throwables;
import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.util.concurrent.UncheckedExecutionException;

import java.io.File;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cache for {@link ExtensionModel}s
 */
public class ExtensionModelServiceCache implements Disposable {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE =
      "tooling.client.ExtensionModelServiceCache.cache";
  private static final Logger LOGGER = LoggerFactory.getLogger(ExtensionModelServiceCache.class);

  private List<ExtensionModel> muleExtensionModels;
  private Cache<ArtifactDescriptor, Optional<Pair<ExtensionModel, ArtifactClassLoader>>> extensionModelsByArtifact =
      CacheBuilder.newBuilder()
          .maximumSize(cacheMaximumSize())
          .removalListener((RemovalListener<ArtifactDescriptor, Optional<Pair<ExtensionModel, ArtifactClassLoader>>>) notification -> notification
              .getValue()
              .ifPresent(pair -> closeClassLoader(pair.getSecond())))
          .build();
  private Cache<File, Optional<Pair<ExtensionModel, ArtifactClassLoader>>> extensionModelsByFile =
      CacheBuilder.newBuilder()
          .maximumSize(cacheMaximumSize())
          .removalListener((RemovalListener<File, Optional<Pair<ExtensionModel, ArtifactClassLoader>>>) notification -> notification
              .getValue()
              .ifPresent(pair -> closeClassLoader(pair.getSecond())))
          .build();

  private Integer cacheMaximumSize() {
    Integer cacheSize = valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE, "100"));
    Preconditions.checkArgument(cacheSize > 0,
                                format("Wrong value %d provided in system property %s, the cache cannot be less that zero",
                                       cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE));
    return cacheSize;
  }

  /**
   * Loads an extension model from a {@link ArtifactDescriptor}.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param pluginDescriptor the plugin descriptor
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the extension model, empty if the plugin does not have an extension model
   */
  public Optional<ExtensionModel> loadExtensionModel(ArtifactDescriptor pluginDescriptor,
                                                     InternalExtensionModelService extensionModelService) {
    try {
      return extensionModelsByArtifact
          .get(pluginDescriptor, () -> extensionModelService.loadPairArtifactClassLoaderAndExtensionModel(pluginDescriptor))
          .map(Pair::getFirst);
    } catch (ExecutionException e) {
      if (e.getCause() instanceof RuntimeException) {
        throw (RuntimeException) e.getCause();
      } else {
        throw new RuntimeException(e.getCause());
      }
    } catch (UncheckedExecutionException e) {
      Throwables.propagateIfPossible(e.getCause());
      throw Throwables.propagate(e);
    }
  }

  /**
   * Loads an extension model from a {@link File} pointing to the plugin artifact.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param plugin the plugin descriptor
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the extension model, empty if the plugin does not have an extension model
   */
  public Optional<ExtensionModel> loadExtensionModel(File plugin, InternalExtensionModelService extensionModelService) {
    try {
      return extensionModelsByFile.get(plugin, () -> extensionModelService.loadPairArtifactClassLoaderAndExtensionModel(plugin))
          .map(Pair::getFirst);
    } catch (ExecutionException e) {
      if (e.getCause() instanceof RuntimeException) {
        throw (RuntimeException) e.getCause();
      } else {
        throw new RuntimeException(e.getCause());
      }
    } catch (UncheckedExecutionException e) {
      Throwables.propagateIfPossible(e.getCause());
      throw Throwables.propagate(e);
    }
  }

  /**
   * Loads the core extension model.
   *
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the core extension model
   */
  public List<ExtensionModel> loadMuleExtensionModels(InternalExtensionModelService extensionModelService) {
    if (muleExtensionModels == null) {
      muleExtensionModels = extensionModelService.loadMuleExtensionModels();
    }
    return muleExtensionModels;
  }

  private void closeClassLoader(ArtifactClassLoader artifactClassLoader) {
    try {
      artifactClassLoader.dispose();
    } catch (Exception e) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug(e.getMessage(), e);
      }
    }
  }

  @Override
  public void dispose() {
    this.extensionModelsByFile.invalidateAll();
    this.extensionModelsByArtifact.invalidateAll();
  }

}
