/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import static org.mule.runtime.api.exception.ExceptionHelper.getRootException;
import static org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse.builder;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.dataweave.validation.DataWeaveValidationError;
import org.mule.tooling.client.api.dataweave.validation.DataWeaveValidationRequest;
import org.mule.tooling.client.api.dataweave.validation.DataWeaveValidationResponse;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.weave.v2.el.WeaveExpressionLanguage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.io.IOUtils;
import org.mule.weave.v2.parser.Message;
import org.mule.weave.v2.parser.ast.structure.DocumentNode;
import org.mule.weave.v2.parser.ast.variables.NameIdentifier;
import org.mule.weave.v2.parser.location.Position;
import org.mule.weave.v2.parser.location.WeaveLocation;
import org.mule.weave.v2.parser.phase.ParsingContext;
import org.mule.weave.v2.parser.phase.PhaseResult;
import org.mule.weave.v2.runtime.CompilationResult;
import org.mule.weave.v2.runtime.WeaveCompiler;
import org.mule.weave.v2.sdk.ParsingContextFactory;
import org.mule.weave.v2.sdk.WeaveResource;
import org.mule.weave.v2.sdk.WeaveResourceFactory;
import scala.Option;
import scala.Tuple2;
import scala.Tuple3;
import scala.collection.JavaConverters;
import scala.collection.Seq;

/**
 * @deprecated keep this version until it is validated and we can get rid of the remote runner and unify LocalRunner with class loader support.
 */
@Deprecated
public class LocalRunner implements DataWeaveRunner {

  private static final String APPLICATION_JAVA = "application/java";
  private static final String DEFAULT_ATTRIBUTES_MEDIA_TYPE = "application/java";

  private static final String VARIABLES_ID = "vars";
  private static final String ATTRIBUTES_ID = "attributes";
  protected static final String PAYLOAD_ID = "payload";
  private static final String ERROR_ID = "error";

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(getExecutionClassLoader());
    try {
      WeaveExpressionLanguage executor = new WeaveExpressionLanguage();
      TypedValue response = executor.evaluate(getScript(parameters), buildContextFromParameters(parameters));

      return builder().result(getResult(response, executor)).mediaType(getMediaType(response)).valid(true).build();
    } catch (Exception e) {
      return builder().errorMessage(getRootException(e).getMessage()).build();
    } finally {
      Thread.currentThread().setContextClassLoader(currentClassLoader);
    }
  }

  protected ClassLoader getExecutionClassLoader() {
    return this.getClass().getClassLoader();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeaveValidationResponse validate(DataWeaveValidationRequest parameters) {
    ClassLoader currentClassLoader = Thread.currentThread().getContextClassLoader();
    Thread.currentThread().setContextClassLoader(getExecutionClassLoader());
    try {
      final ParsingContext parsingContext = createParsingContext();

      final String script = parameters.getScript();
      final WeaveResource resource = WeaveResourceFactory.fromContent(script);

      final PhaseResult<CompilationResult<DocumentNode>> compilationResult = WeaveCompiler.compile(resource, parsingContext);

      final Seq<Tuple2<WeaveLocation, Message>> errorMessages = compilationResult.errorMessages();

      return DataWeaveValidationResponse.builder().withErrors(fromErrorMessages(errorMessages)).build();
    } catch (final Exception e) {
      return DataWeaveValidationResponse.builder().withException(e).build();
    } finally {
      Thread.currentThread().setContextClassLoader(currentClassLoader);
    }
  }

  private List<DataWeaveValidationError> fromErrorMessages(final Seq<Tuple2<WeaveLocation, Message>> errorMessages) {
    return JavaConverters.seqAsJavaList(errorMessages).stream()
        .map(this::fromErrorTuple)
        .collect(Collectors.toList());
  }

  private DataWeaveValidationError fromErrorTuple(final Tuple2<WeaveLocation, Message> errorTuple) {
    final Option<Tuple3<Position, Position, NameIdentifier>> maybeLocation = WeaveLocation.unapply(errorTuple._1);
    if (maybeLocation.isEmpty()) {
      return null;
    }

    final Tuple3<Position, Position, NameIdentifier> location = maybeLocation.get();

    final Position startPosition = location._1();
    final Position endPosition = location._2();

    final Message message = errorTuple._2;

    return DataWeaveValidationError.builder()
        .withStartIndex(startPosition.index())
        .withStartLine(startPosition.line())
        .withEndIndex(endPosition.index())
        .withEndLine(endPosition.line())
        .withMessage(message.message())
        .build();
  }

  private ParsingContext createParsingContext() {
    final ParsingContext parsingContext = ParsingContextFactory.createMappingParsingContext(NameIdentifier.anonymous());

    parsingContext.addImplicitInput(PAYLOAD_ID, Option.empty());
    parsingContext.addImplicitInput(ATTRIBUTES_ID, Option.empty());
    parsingContext.addImplicitInput(VARIABLES_ID, Option.empty());
    parsingContext.addImplicitInput(ERROR_ID, Option.empty());

    return parsingContext;
  }

  private String getMediaType(TypedValue evaluate) {
    DataType dataType = evaluate.getDataType();
    MediaType mediaType = dataType.getMediaType();
    return mediaType.withoutParameters().toString();
  }

  private String getResult(TypedValue result, WeaveExpressionLanguage executor) {
    final Object value = result.getValue();
    if (value instanceof CursorStreamProvider) {
      return manageCursorStream(result);
    } else {
      return serializeObject(value, executor);
    }
  }

  protected String manageCursorStream(TypedValue<CursorStreamProvider> result) {
    CursorStreamProvider streamProvider = null;
    try {
      streamProvider = result.getValue();
      Charset encoding = result.getDataType().getMediaType().getCharset().orElseGet(Charset::defaultCharset);
      try (CursorStream cursorStream = streamProvider.openCursor()) {
        return IOUtils.toString(cursorStream, encoding);
      }
    } catch (IOException e) {
      throw new ToolingException("DataWeave :: Error consuming DataWeave result", e);
    } finally {
      if (streamProvider != null) {
        streamProvider.close();
      }
    }
  }

  protected String serializeObject(Object value, WeaveExpressionLanguage executor) {
    Gson gson = new GsonBuilder().setPrettyPrinting().create();
    return gson.toJson(value);
  }

  private BindingContext buildContextFromParameters(DataWeavePreviewRequest parameters) throws IOException {
    BindingContext.Builder bindingBuilder = BindingContext.builder();

    final EventModel event = getEvent(parameters);
    if (event != null) {
      manageVariables(bindingBuilder, event);
      manageMessage(bindingBuilder, event);
    }

    return bindingBuilder.build();
  }

  private void manageMessage(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    final MessageModel message = event.getMessage();
    if (message != null) {
      if (message.getAttributes() != null) {
        bindingBuilder.addBinding(ATTRIBUTES_ID, asTypedValue(message.getAttributes()));
      }

      if (message.getPayload() != null) {
        bindingBuilder.addBinding(PAYLOAD_ID, asTypedValue(message.getPayload()));
      }
    }
  }

  private void manageVariables(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    if (event.getVariables() != null && !event.getVariables().isEmpty()) {
      Map<String, TypedValue> variables = new HashMap<>();
      for (Map.Entry<String, TypedValueModel> pair : event.getVariables().entrySet()) {
        variables.put(pair.getKey(), asTypedValue(pair.getValue()));
      }
      DataType variablesDataType = DataType.builder().type(Map.class).mediaType(DEFAULT_ATTRIBUTES_MEDIA_TYPE).build();
      bindingBuilder.addBinding(VARIABLES_ID, new TypedValue<>(variables, variablesDataType));
    }
  }

  protected TypedValue<?> asTypedValue(TypedValueModel restTypedValue) throws IOException {
    String mediaType = restTypedValue.getDataType().getMediaType();
    if (APPLICATION_JAVA.equals(mediaType)) {
      throw new IllegalArgumentException("Java input not supported, serialize to DW script");
    } else {
      DataType dataType = DataType.builder().type(String.class).mediaType(mediaType).build();
      return new TypedValue<>(new String(restTypedValue.getContent(), "UTF-8"), dataType);
    }
  }

  private EventModel getEvent(DataWeavePreviewRequest parameters) {
    return parameters.getEvent();
  }

  private String getScript(DataWeavePreviewRequest parameters) {
    return parameters.getScript();
  }
}
