/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static org.mule.runtime.config.spring.api.XmlConfigurationDocumentLoader.noValidationDocumentLoader;
import static org.mule.runtime.config.spring.internal.util.ComponentBuildingDefinitionUtils.registerComponentBuildingDefinitions;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.runtime.core.api.util.collection.Collectors.toImmutableList;
import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.config.spring.api.dsl.model.ApplicationModel;
import org.mule.runtime.config.spring.api.dsl.model.ComponentBuildingDefinitionRegistry;
import org.mule.runtime.config.spring.api.dsl.processor.ArtifactConfig;
import org.mule.runtime.config.spring.api.dsl.processor.ConfigFile;
import org.mule.runtime.config.spring.api.dsl.processor.ConfigLine;
import org.mule.runtime.config.spring.api.dsl.processor.xml.XmlApplicationParser;
import org.mule.runtime.config.spring.api.dsl.xml.StaticXmlNamespaceInfo;
import org.mule.runtime.config.spring.api.dsl.xml.StaticXmlNamespaceInfoProvider;
import org.mule.runtime.config.spring.internal.dsl.model.ClassLoaderResourceProvider;
import org.mule.runtime.core.api.registry.ServiceRegistry;
import org.mule.runtime.core.api.registry.SpiServiceRegistry;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfo;
import org.mule.runtime.dsl.api.xml.XmlNamespaceInfoProvider;
import org.mule.tooling.client.api.exception.ToolingException;

import com.google.common.collect.ImmutableList;

import java.io.InputStream;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import org.w3c.dom.Document;

/**
 * Default implementation for {@link ApplicationModelFactory}.
 *
 * @since 4.0
 */
public class DefaultApplicationModelFactory implements ApplicationModelFactory {

  private ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader;

  /**
   * Creates a new instance.
   *
   * @param componentBuildingDefinitionLoader the loader to use for getting the
   *        {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinition}s from a
   *        {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinitionProvider}.
   */
  public DefaultApplicationModelFactory(ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    this.componentBuildingDefinitionLoader = componentBuildingDefinitionLoader;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ApplicationModel> createApplicationModel(ApplicationDescriptor applicationDescriptor,
                                                           Set<ExtensionModel> extensionModels,
                                                           ClassLoader applicationClassLoader) {
    try {
      return of(loadApplicationModel(applicationDescriptor.getConfigResources().stream()
          .map(configFile -> {
            InputStream configAsStream = applicationClassLoader.getResourceAsStream(configFile);
            if (configAsStream == null) {
              throw new ToolingException(format("Configuration file '%s' cannot be found on application class loader",
                                                configFile));
            }
            return loadConfigFile(applicationDescriptor, configFile, configAsStream, extensionModels)
                .get();
          })
          .collect(toList()), extensionModels, applicationClassLoader));
    } catch (Exception e) {
      throw new ToolingException(e);
    }
  }

  /**
   * Loads {@link ConfigFile} from the configuration file {@link InputStream}.
   *
   * @param applicationDescriptor {@link ApplicationDescriptor} for the application.
   * @param fileName the name of the config file.
   * @param inputStream {@link InputStream} content of the file.
   * @param extensionModels {@link List} of {@link ExtensionModel} discovered from the application.
   * @return a optional {@link ConfigFile}.
   */
  private Optional<ConfigFile> loadConfigFile(ApplicationDescriptor applicationDescriptor, String fileName,
                                              InputStream inputStream, Set<ExtensionModel> extensionModels) {
    return withContextClassLoader(this.getClass().getClassLoader(), () -> {
      List<XmlNamespaceInfoProvider> xmlNamespaceInfoProviders =
          ImmutableList.<XmlNamespaceInfoProvider>builder()
              .add(createStaticNamespaceInfoProviders(extensionModels))
              .addAll(discoverRuntimeXmlNamespaceInfoProvider())
              .build();
      XmlApplicationParser xmlApplicationParser = new XmlApplicationParser(xmlNamespaceInfoProviders);
      Document document = noValidationDocumentLoader().loadDocument(fileName, inputStream);
      return xmlApplicationParser.parse(document.getDocumentElement())
          .map(configLine -> new ConfigFile(relativizeConfigFileName(applicationDescriptor, fileName),
                                            singletonList(configLine)));
    });
  }

  private List<XmlNamespaceInfoProvider> discoverRuntimeXmlNamespaceInfoProvider() {
    ImmutableList.Builder namespaceInfoProvidersBuilder = ImmutableList.builder();
    namespaceInfoProvidersBuilder
        .addAll(new SpiServiceRegistry().lookupProviders(XmlNamespaceInfoProvider.class, this.getClass().getClassLoader()));
    return namespaceInfoProvidersBuilder.build();
  }

  private XmlNamespaceInfoProvider createStaticNamespaceInfoProviders(Set<ExtensionModel> extensionModels) {
    List<XmlNamespaceInfo> extensionNamespaces = extensionModels.stream()
        .map(ext -> new StaticXmlNamespaceInfo(ext.getXmlDslModel().getNamespace(), ext.getXmlDslModel().getPrefix()))
        .collect(toImmutableList());

    return new StaticXmlNamespaceInfoProvider(extensionNamespaces);
  }

  private String relativizeConfigFileName(ApplicationDescriptor applicationDescriptor, String fileName) {
    if (fileName == null) {
      return null;
    }
    return applicationDescriptor.getConfigResources().stream()
        .filter(relativeConfigFileName -> relativeConfigFileName.endsWith(fileName)).findFirst().orElse(null);
  }

  /**
   * Loads {@link ApplicationModel} from {@link ConfigLine}.
   *
   * @param configFiles {@link List configFiles} to be added to the {@link ApplicationModel}.
   * @param extensionModels {@link Set} of {@link ExtensionModel extension models} to be used for generating the application
   *        model.
   * @param applicationClassLoader {@link ClassLoader} to resolve resources.
   * @return a {@link ApplicationModel}.
   */
  private ApplicationModel loadApplicationModel(List<ConfigFile> configFiles, Set<ExtensionModel> extensionModels,
                                                ClassLoader applicationClassLoader)
      throws Exception {
    final ArtifactConfig.Builder builder = new ArtifactConfig.Builder();
    if (!configFiles.isEmpty()) {
      configFiles.forEach(configFile -> builder.addConfigFile(configFile));
    }
    ArtifactConfig artifactConfig = builder.build();
    ApplicationModel applicationModel =
        new ApplicationModel(artifactConfig, null, extensionModels, emptyMap(), of(new EmptyConfigurationProperties()),
                             of(createComponentBuildingDefinitionRegistry(extensionModels,
                                                                          componentBuildingDefinitionLoader)),
                             false, new ClassLoaderResourceProvider(applicationClassLoader));
    return applicationModel;
  }

  private static ComponentBuildingDefinitionRegistry createComponentBuildingDefinitionRegistry(
                                                                                               Set<ExtensionModel> extensionModels,
                                                                                               ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader) {
    ServiceRegistry serviceRegistry = new SpiServiceRegistry();
    final ComponentBuildingDefinitionRegistry componentBuildingDefinitionRegistry =
        new ComponentBuildingDefinitionRegistry();

    registerComponentBuildingDefinitions(serviceRegistry, DefaultApplicationModelFactory.class.getClassLoader(),
                                         componentBuildingDefinitionRegistry, ofNullable(extensionModels),
                                         componentBuildingDefinitionLoader.loadComponentBuildingDefinitions(extensionModels));

    return componentBuildingDefinitionRegistry;
  }

  /**
   * Implementation that allows to load an {@link ApplicationModel} without having the properties replaced.
   */
  private class EmptyConfigurationProperties implements ConfigurationProperties {

    @Override
    public <T> Optional<T> resolveProperty(String s) {
      return of((T) "");
    }

    @Override
    public Optional<Boolean> resolveBooleanProperty(String s) {
      return of(false);
    }

    @Override
    public Optional<String> resolveStringProperty(String s) {
      return of("");
    }

  }
}
