/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static java.util.UUID.randomUUID;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.util.LazyValue;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.tryit.TryItService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.ApplicationService;
import org.mule.tooling.client.internal.application.DefaultApplication;
import org.mule.tooling.client.internal.artifact.DefaultArtifactSerializationService;

import java.io.InputStream;
import java.net.URL;
import java.util.Optional;

/**
 * Implementation of a {@link ToolingRuntimeClient}. It uses SPI to look up for {@link RuntimeToolingService} implementation to
 * resolve tooling operations.
 *
 * @since 4.0
 */
class DefaultToolingRuntimeClient implements ToolingRuntimeClient {

  private DefaultToolingArtifactContext context;

  private ArtifactSerializationService artifactSerializationService;
  private TryItService tryItService;
  private ExtensionModelService extensionModelService;

  /**
   * Creates an instance of the client.
   *
   * @param mavenClient {@link MavenClient} to resolve dependencies. Non null.
   * @param agentConfigurationOptional {@link Optional} Mule Agent configuration. Non null.
   * @param muleRuntimeExtensionModelProvider provider to load {@link org.mule.runtime.api.meta.model.ExtensionModel}s. Non null.
   * @param componentBuildingDefinitionLoader loader for {@link org.mule.runtime.dsl.api.component.ComponentBuildingDefinition}s. Non null.
   * @param applicationService {@link ApplicationService} to create the {@link org.mule.tooling.client.internal.application.Application} class loader. Non null.
   * @param applicationCache {@link ApplicationCache} to fetch or create {@link org.mule.tooling.client.internal.application.DefaultApplication}. Non null.
   */
  DefaultToolingRuntimeClient(MavenClient mavenClient,
                              Optional<AgentConfiguration> agentConfigurationOptional,
                              MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                              ComponentBuildingDefinitionLoader componentBuildingDefinitionLoader,
                              ApplicationService applicationService, ApplicationCache applicationCache,
                              Optional<InputStream> logConfig) {
    checkNotNull(mavenClient, "aetherDependencyResolver cannot be null");
    checkNotNull(agentConfigurationOptional, "agentConfigurationOptional cannot be null");
    checkNotNull(muleRuntimeExtensionModelProvider, "muleRuntimeExtensionModelProvider cannot be null");
    checkNotNull(applicationService, "applicationService cannot be null");
    checkNotNull(applicationCache, "applicationCache cannot be null");

    context = new DefaultToolingArtifactContext();
    logConfig.ifPresent(config -> context.setLoggingConfiguration(config));
    context.setAgentConfiguration(agentConfigurationOptional);
    context.setMavenClient(mavenClient);
    context.setMuleRuntimeExtensionModelProvider(muleRuntimeExtensionModelProvider);
    context.setComponentBuildingDefinitionLoader(componentBuildingDefinitionLoader);
    context.setApplicationService(applicationService);
    context.setApplicationCache(applicationCache);

    artifactSerializationService =
        new DefaultArtifactSerializationService(muleRuntimeExtensionModelProvider.getMuleExtensionModels(),
                                                muleRuntimeExtensionModelProvider,
                                                mavenClient);
    tryItService = new DefaultTryItService(new LazyValue<>(() -> context.getRuntimeToolingService()));
    extensionModelService = new ToolingExtensionModelAdapter(muleRuntimeExtensionModelProvider);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public TryItService tryItService() {
    return tryItService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExtensionModelService extensionModelService() {
    return extensionModelService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactSerializationService artifactSerializationService() {
    return artifactSerializationService;
  }

  /**
   * {@inheritDoc}
   * @return
   */
  @Override
  public ToolingArtifact newToolingArtifact(URL applicationUrlContent) {
    String id = randomUUID().toString();
    ApplicationCache applicationCache = context.getApplicationCache();
    Application application = applicationCache.getApplication(id, () -> new DefaultApplication(applicationUrlContent, context));
    return newToolingArtifact(id, applicationCache, application);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException {
    Application application = context.getApplicationCache().getApplication(id, () -> {
      throw new ToolingArtifactNotFoundException(
                                                 format("ToolingArtifact not found in cache for id: %s", id));
    });
    application.setContext(context);
    return newToolingArtifact(id, context.getApplicationCache(), application);
  }

  private ToolingArtifact newToolingArtifact(final String id, final ApplicationCache applicationCache,
                                             final Application application) {
    return new ToolingArtifactWrapper(new DefaultToolingArtifact(id, application) {

      @Override
      public void dispose() {
        applicationCache.invalidate(id);
      }
    });
  }

}
