/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.sort;
import static java.util.Comparator.comparingLong;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import org.mule.runtime.api.util.LazyValue;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.agent.rest.client.tooling.applications.applicationName.tryIt.AgentTrackingNotificationResponse;
import org.mule.tooling.client.api.tryit.MessageHistory;
import org.mule.tooling.client.api.tryit.TryItService;
import org.mule.tooling.client.api.types.Transaction;
import org.mule.tooling.client.api.types.TransactionStackEntry;
import org.mule.tooling.client.api.types.TransactionStatus;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation for {@link org.mule.tooling.client.api.tryit.TryItService}.
 *
 * @since 4.0
 */
public class DefaultTryItService implements TryItService {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String FIRST_MESSAGE_PROCESSOR_INDEX = "/0";
  private static final String MESSAGE_PRE_INVOKE_ACTION = "message processor pre invoke";
  private static final String MESSAGE_POST_INVOKE_ACTION = "message processor post invoke";

  private LazyValue<RuntimeToolingService> runtimeToolingServiceLazyValue;

  /**
   * Creates a default instance of the service.
   *
   * @param runtimeToolingServiceLazyValue {@link LazyValue} for {@link RuntimeToolingService}.
   */
  protected DefaultTryItService(LazyValue<RuntimeToolingService> runtimeToolingServiceLazyValue) {
    requireNonNull(runtimeToolingServiceLazyValue, "runtimeToolingServiceLazyValue cannot be null");

    this.runtimeToolingServiceLazyValue = runtimeToolingServiceLazyValue;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void enable(String applicationName) {
    runtimeToolingServiceLazyValue.get().enableTryIt(applicationName);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageHistory consume(String applicationName, int chunkSize) {
    List<AgentTrackingNotificationResponse> agentTrackingNotifications =
        runtimeToolingServiceLazyValue.get().consumeTryItNotifications(applicationName, chunkSize);
    return convertToMessageHistory(agentTrackingNotifications);
  }

  private MessageHistory convertToMessageHistory(List<AgentTrackingNotificationResponse> agentTrackingNotifications) {
    MessageHistory messageHistory = new MessageHistory();
    List<Transaction> transactions = new ArrayList<>();
    messageHistory.setTransactions(transactions);
    Map<String, List<AgentTrackingNotificationResponse>> notificationsMap =
        splitAgentNotificationsAndSortByDate(agentTrackingNotifications);
    notificationsMap.keySet().stream().forEach(transactionId -> {
      List<AgentTrackingNotificationResponse> transactionNotifications = notificationsMap.get(transactionId);
      Transaction transaction = new Transaction();
      // TODO MULE-12229 Support TransactionStatus
      transaction.setTransactionStatus(TransactionStatus.COMPLETE);
      transaction.setId(transactionNotifications.get(0).getTransactionId());
      AgentTrackingNotificationResponse firstMessageProcessorInputNotification = transactionNotifications.get(0);
      if (firstMessageProcessorInputNotification.getComponentLocation().getLocation().endsWith(FIRST_MESSAGE_PROCESSOR_INDEX) &&
          firstMessageProcessorInputNotification.getAction().equals(MESSAGE_PRE_INVOKE_ACTION)) {
        transaction.setMessage(firstMessageProcessorInputNotification.getEvent().getMessage());
        transaction.setTimestamp(firstMessageProcessorInputNotification.getTimestamp());
      }
      transaction.setTransactionStack(transactionNotifications.stream()
          .filter(agentTrackingNotification -> agentTrackingNotification.getAction().equals(MESSAGE_POST_INVOKE_ACTION))
          .map(agentTrackingNotification -> {
            TransactionStackEntry transactionStackEntry = new TransactionStackEntry();
            transactionStackEntry.setTimestamp(agentTrackingNotification.getTimestamp());
            transactionStackEntry.setComponentLocation(agentTrackingNotification.getComponentLocation());
            transactionStackEntry.setEventModel(agentTrackingNotification.getEvent());
            return transactionStackEntry;
          }).collect(toList()));
      transactions.add(transaction);
    });
    return messageHistory;
  }

  private Map<String, List<AgentTrackingNotificationResponse>> splitAgentNotificationsAndSortByDate(List<AgentTrackingNotificationResponse> agentTrackingNotifications) {
    logger.debug("Grouping notifications by correlationId");
    Map<String, List<AgentTrackingNotificationResponse>> notificationsMap = new HashMap<>();
    agentTrackingNotifications.stream().forEach(agentTrackingNotification -> {
      logger.debug("Processing notification: {}", agentTrackingNotification);
      List<AgentTrackingNotificationResponse> notifications = notificationsMap.get(agentTrackingNotification.getCorrelationId());
      if (notifications == null) {
        notifications = new ArrayList<>();
        notificationsMap.put(agentTrackingNotification.getCorrelationId(), notifications);
      }
      notifications.add(agentTrackingNotification);
    });

    for (List<AgentTrackingNotificationResponse> transactinAgentNotifications : notificationsMap.values()) {
      sort(transactinAgentNotifications, comparingLong(AgentTrackingNotificationResponse::getTimestamp));
    }
    return notificationsMap;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void disable(String applicationName) {
    try {
      runtimeToolingServiceLazyValue.get().disableTryIt(applicationName);
    } catch (Exception e) {
      logger.warn("Error while disabling application for try it", e);
    }
  }

}
