/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Throwables.propagateIfPossible;
import static java.lang.String.format;
import static java.util.concurrent.CompletableFuture.supplyAsync;
import static java.util.concurrent.TimeUnit.MILLISECONDS;
import org.mule.tooling.client.api.exception.TimeoutException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.request.AbstractToolingRequest;

import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Utility that allows to execute an {@link AbstractToolingRequest} operation and abort if the configuration timeout
 * is reached before the operation is completed.
 *
 * @since 4.0
 */
public final class TimeoutMethodUtils {

  private static final Logger LOGGER = LoggerFactory.getLogger(TimeoutMethodUtils.class);

  /**
   * Evalues the resultSupplier provided for the given timeout defined for the request, if the operation
   * is not completed before timeout is reached a {@link TimeoutException} is thrown. In case of errors during the operation
   * a function is provided for handling them properly.
   *
   * @param timeout the request operation timeout in milliseconds. -1 means no timeout block until it gets the response.
   * @param resultSupplier the function that evaluates the operation and returns the result. Not null.
   * @param throwableTFunction in case if the operation throws an exception an optional function to handle it can be passed.
   * @param cleanupFunctionOnTimeout in case if the operation is timed out an optional function to clean up resources is executed.
   * @param <T> the type of the operation result.
   * @return the result of evaluating the operation.
   */
  public static <T> T withTimeout(long timeout, Supplier<T> resultSupplier,
                                  Optional<Function<Throwable, T>> throwableTFunction,
                                  Optional<Consumer<java.util.concurrent.TimeoutException>> cleanupFunctionOnTimeout) {
    checkNotNull(timeout, "request cannot be null");
    checkNotNull(resultSupplier, "resultSupplier cannot be null");

    CompletableFuture<T> completableFuture = supplyAsync(resultSupplier);
    if (throwableTFunction.isPresent()) {
      completableFuture = completableFuture.exceptionally(throwableTFunction.get());
    }
    try {
      if (timeout != -1) {
        return completableFuture.get(timeout, MILLISECONDS);
      }
      return completableFuture.get();
    } catch (java.util.concurrent.TimeoutException e) {
      if (cleanupFunctionOnTimeout.isPresent()) {
        try {
          cleanupFunctionOnTimeout.get().accept(e);
        } catch (Exception internalError) {
          LOGGER.warn("Error while calling function to clean up resources on a timeout", internalError);
        }
      }
      throw new TimeoutException(format("Couldn't resolve the operation in the the expected time frame (%sms)", timeout), e);
    } catch (InterruptedException | ExecutionException e) {
      propagateIfPossible(e.getCause());
      throw new ToolingException(e);
    }
  }

}
