/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.stream.Collectors.toList;

import org.mule.runtime.extension.api.dsl.syntax.resources.spi.ExtensionSchemaGenerator;
import org.mule.runtime.extension.api.persistence.ExtensionModelJsonSerializer;
import org.mule.runtime.module.extension.internal.capability.xml.schema.DefaultExtensionSchemaGenerator;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import java.io.File;
import java.util.List;
import java.util.Optional;

/**
 * Implementation of {@link ExtensionModelService} that deleages to a {@link MuleRuntimeExtensionModelProvider} and builds the
 * Tooling view of a Mule Runtime {@link org.mule.runtime.api.meta.model.ExtensionModel} to {@link org.mule.tooling.client.api.extension.model.ExtensionModel}.
 *
 * @since 4.0
 */
public class ToolingExtensionModelAdapter implements ExtensionModelService {

  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  private ExtensionSchemaGenerator extensionSchemaGenerator = new DefaultExtensionSchemaGenerator();


  public ToolingExtensionModelAdapter(MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider) {
    this.muleRuntimeExtensionModelProvider = muleRuntimeExtensionModelProvider;
  }

  @Override
  public Optional<ExtensionModel> loadExtensionModel(ArtifactDescriptor pluginDescriptor) {
    return muleRuntimeExtensionModelProvider.getExtensionModel(pluginDescriptor)
        .map(runtimeExtensionModel -> toDTO(runtimeExtensionModel));
  }

  @Override
  public Optional<ExtensionModel> loadExtensionModel(File plugin) {
    return muleRuntimeExtensionModelProvider.getExtensionModel(plugin).map(runtimeExtensionModel -> toDTO(runtimeExtensionModel));
  }

  @Override
  public Optional<String> loadExtensionSchema(ArtifactDescriptor pluginDescriptor) {
    return muleRuntimeExtensionModelProvider.getExtensionSchema(pluginDescriptor);
  }

  @Override
  public Optional<String> loadExtensionSchema(File plugin) {
    return muleRuntimeExtensionModelProvider.getExtensionSchema(plugin);
  }

  @Override
  public List<ExtensionModel> loadMuleExtensionModels() {
    return muleRuntimeExtensionModelProvider.getMuleExtensionModels().stream()
        .map(runtimeExtensionModel -> toDTO(runtimeExtensionModel))
        .collect(toList());
  }

  //TODO (gfernandes) MULE-13103
  @Override
  public Optional<org.mule.runtime.api.meta.model.ExtensionModel> internalExtensionModel(File plugin) {
    return muleRuntimeExtensionModelProvider.getExtensionModel(plugin);
  }

  //TODO (gfernandes) MULE-13103
  @Override
  public List<org.mule.runtime.api.meta.model.ExtensionModel> internalMuleExtensionModels() {
    return muleRuntimeExtensionModelProvider.getMuleExtensionModels();
  }

  private ExtensionModel toDTO(org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel) {
    ExtensionModelJsonSerializer serializer = new ExtensionModelJsonSerializer();
    org.mule.runtime.api.meta.model.ExtensionModel serializedExtensionModel =
        serializer.deserialize(serializer.serialize(runtimeExtensionModel));
    return new ExtensionModelFactory().createExtensionModel(serializedExtensionModel);
  }

}
