/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Throwables.propagate;
import static com.google.common.base.Throwables.propagateIfPossible;
import static java.lang.System.getProperty;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.internal.application.Application;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.RemovalListener;
import com.google.common.util.concurrent.UncheckedExecutionException;

import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cache for {@link Application}.
 */
public class ApplicationCache implements Disposable {

  private static final String TOOLING_CLIENT_APPLICATION_CACHE_EXPIRE_AFTER_ACCESS =
      "tooling.client.ApplicationCache.expireAfterAccess";
  private static final Logger LOGGER = LoggerFactory.getLogger(ApplicationCache.class);

  private Cache<String, Application> applicationsById =
      CacheBuilder.newBuilder()
          .expireAfterAccess(Long.valueOf(getProperty(TOOLING_CLIENT_APPLICATION_CACHE_EXPIRE_AFTER_ACCESS, "300")),
                             TimeUnit.SECONDS)
          .removalListener((RemovalListener<String, Application>) notification -> notification
              .getValue().dispose())
          .build();

  public ApplicationCache() {
    LOGGER.info("Initialising Application cache");
  }

  /**
   * Gets the {@link Application} for the given id.
   * <p>
   * It uses a cache, so it will only call the {@code callable} if there's a cache miss.
   *
   * @param id the id of the {@link Application }to be retrieved.
   * @param callable {@link Callable} to create a {@link Application} in case of a cache miss.
   * @return the {@link Application}
   */
  public Application getApplication(String id,
                                    Callable<Application> callable) {
    try {
      return applicationsById.get(id, callable);
    } catch (ExecutionException e) {
      if (e.getCause() instanceof RuntimeException) {
        throw (RuntimeException) e.getCause();
      } else {
        throw new RuntimeException(e.getCause());
      }
    } catch (UncheckedExecutionException e) {
      propagateIfPossible(e.getCause());
      throw propagate(e);
    }
  }

  public void invalidate(String id) {
    applicationsById.invalidate(id);
  }

  @Override
  public void dispose() {
    LOGGER.info("Disposing Application cache");
    this.applicationsById.invalidateAll();
  }

}
