/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Optional.empty;
import static java.util.Optional.ofNullable;
import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.serialization.SerializationUtilsImpl.deserialize;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.MavenConfiguration;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClient.Builder;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.internal.application.ApplicationService;
import org.mule.tooling.client.internal.application.DefaultApplicationService;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;

import java.util.Optional;

/**
 * Default implementation of {@link Builder} that creates a {@link ToolingRuntimeClient}.
 *
 * @since 4.0
 */
public class DefaultToolingRuntimeClientBuilder implements Builder, Command {

  private MavenClient mavenClient;
  private ModuleRepository moduleRepository;
  private ArtifactClassLoader containerClassLoaderFactory;
  private ExtensionModelServiceCache extensionModelServiceCache;
  private ApplicationCache applicationCache;
  private Optional<MetadataCacheFactory> metadataCacheFactoryOptional = empty();
  private DslSyntaxServiceCache dslSyntaxServiceCache;

  private AgentConfiguration agentConfiguration;

  public DefaultToolingRuntimeClientBuilder(ModuleRepository moduleRepository,
                                            ArtifactClassLoader containerClassLoaderFactory,
                                            ExtensionModelServiceCache extensionModelServiceCache,
                                            ApplicationCache applicationCache,
                                            DslSyntaxServiceCache dslSyntaxServiceCache) {
    this.moduleRepository = moduleRepository;
    this.containerClassLoaderFactory = containerClassLoaderFactory;
    this.extensionModelServiceCache = extensionModelServiceCache;
    this.applicationCache = applicationCache;
    this.dslSyntaxServiceCache = dslSyntaxServiceCache;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Builder withRemoteAgentConfiguration(AgentConfiguration agentConfiguration) {
    this.agentConfiguration = agentConfiguration;
    return this;
  }

  @Override
  public Builder withMavenConfiguration(MavenConfiguration mavenConfiguration) {
    this.mavenClient = discoverProvider(DefaultToolingRuntimeClientBuilder.class.getClassLoader())
        .createMavenClient(mavenConfiguration);
    return this;
  }

  @Override
  public Builder withMetadataCacheFactory(MetadataCacheFactory metadataCacheFactory) {
    this.metadataCacheFactoryOptional = ofNullable(metadataCacheFactory);
    return this;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingRuntimeClient build() {
    MuleArtifactResourcesRegistry muleArtifactResourcesRegistry =
        new MuleArtifactResourcesRegistry(mavenClient, moduleRepository,
                                          containerClassLoaderFactory);
    InternalExtensionModelService extensionModelService =
        new DefaultExtensionModelService(mavenClient, muleArtifactResourcesRegistry);
    MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider =
        new CachedExtensionModelService(extensionModelServiceCache,
                                        extensionModelService);
    ApplicationService applicationService = new DefaultApplicationService(muleArtifactResourcesRegistry);

    return new DefaultToolingRuntimeClient(mavenClient,
                                           ofNullable(agentConfiguration), muleRuntimeExtensionModelProvider,
                                           applicationService, applicationCache,
                                           metadataCacheFactoryOptional, dslSyntaxServiceCache);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "withRemoteAgentConfiguration": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(AgentConfiguration.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        this.withRemoteAgentConfiguration(deserialize(arguments[0]));
        return this;
      }
      case "withMavenConfiguration": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(MavenConfiguration.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        this.withMavenConfiguration(deserialize(arguments[0]));
        return this;
      }
      case "build": {
        return this.build();
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }
}
