/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static org.apache.commons.lang3.StringUtils.isEmpty;
import static org.mule.runtime.container.internal.ClasspathModuleDiscoverer.EXPORTED_CLASS_PACKAGES_PROPERTY;
import static org.mule.runtime.container.internal.ClasspathModuleDiscoverer.EXPORTED_RESOURCE_PROPERTY;
import static org.mule.runtime.container.internal.ClasspathModuleDiscoverer.EXPORTED_SERVICES_PROPERTY;
import static org.mule.runtime.container.internal.ClasspathModuleDiscoverer.MODULE_PROPERTIES;
import static org.mule.runtime.container.internal.ClasspathModuleDiscoverer.PRIVILEGED_ARTIFACTS_PROPERTY;
import static org.mule.runtime.container.internal.ClasspathModuleDiscoverer.PRIVILEGED_EXPORTED_CLASS_PACKAGES_PROPERTY;
import static org.mule.runtime.core.api.util.PropertiesUtils.discoverProperties;
import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.container.api.MuleModule;
import org.mule.runtime.container.internal.CompositeModuleDiscoverer;
import org.mule.runtime.container.internal.ContainerClassLoaderFactory;
import org.mule.runtime.container.internal.JreModuleDiscoverer;
import org.mule.runtime.container.internal.ModuleDiscoverer;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ExportedService;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.ToolingRuntimeClientBuilderFactory;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;
import org.mule.tooling.client.internal.log4j.ToolingLog4jContextFactory;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Properties;
import java.util.Set;

import org.apache.logging.log4j.LogManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link ToolingRuntimeClientBuilderFactory} for creating {@link ToolingRuntimeClient}s.
 *
 * @since 4.0
 */
public class DefaultToolingRuntimeClientFactory implements ToolingRuntimeClientBuilderFactory, Command, Disposable {

  private final ArtifactClassLoader containerClassLoaderFactory;
  private final ModuleRepository moduleRepository;
  private ExtensionModelServiceCache extensionModelServiceCache;
  private ApplicationCache applicationCache;
  private DslSyntaxServiceCache dslSyntaxServiceCache;

  public DefaultToolingRuntimeClientFactory() {
    extensionModelServiceCache = new ExtensionModelServiceCache();
    moduleRepository = createModuleRepository();
    containerClassLoaderFactory = createContainerClassLoader(moduleRepository);
    applicationCache = new ApplicationCache();
    dslSyntaxServiceCache = new DslSyntaxServiceCache();
  }

  public static ArtifactClassLoader createContainerClassLoader(ModuleRepository moduleRepository) {
    ArtifactClassLoader containerClassLoaderFactory;
    containerClassLoaderFactory =
        new ContainerClassLoaderFactory(moduleRepository)
            .createContainerClassLoader(DefaultToolingRuntimeClientFactory.class.getClassLoader());
    return containerClassLoaderFactory;
  }

  public static ModuleRepository createModuleRepository() {
    ModuleDiscoverer jreModuleDiscoverer = new ModuleDiscoverer() {

      List<MuleModule> muleModules;

      @Override
      public List<MuleModule> discover() {
        // cache the result since it's expensive to calculate
        if (muleModules == null) {
          muleModules = new JreModuleDiscoverer().discover();
        }
        return muleModules;
      }
    };
    ModuleDiscoverer moduleDiscoverer = new CompositeModuleDiscoverer(
                                                                      new ModuleDiscoverer[] {jreModuleDiscoverer,
                                                                          new DuplicateClasspathModuleDiscoverer(DefaultToolingRuntimeClientFactory.class
                                                                              .getClassLoader())});
    return () -> moduleDiscoverer.discover();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingRuntimeClient.Builder create() {
    return new DefaultToolingRuntimeClientBuilder(moduleRepository,
                                                  containerClassLoaderFactory,
                                                  extensionModelServiceCache,
                                                  applicationCache,
                                                  dslSyntaxServiceCache);
  }

  @Override
  public void dispose() {
    dispose(extensionModelServiceCache);
    dispose(applicationCache);
    dispose(dslSyntaxServiceCache);

    if (LogManager.getFactory() instanceof ToolingLog4jContextFactory) {
      ((ToolingLog4jContextFactory) LogManager.getFactory()).dispose();
    }
  }

  private void dispose(Disposable disposable) {
    try {
      disposable.dispose();
    } catch (Exception e) {
      // Nothing to do...
    }
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "dispose": {
        this.dispose();
        return null;
      }
      case "create": {
        return create();
      }
    }
    throw new IllegalArgumentException(format("Invalid arguments passed for calling method '%s' on '%s'", methodName,
                                              this.getClass()));
  }

  /**
   * Discoverer for Mule Modules that supports to load duplicate resources, workaround for Equinox issue.
   */
  private static class DuplicateClasspathModuleDiscoverer implements ModuleDiscoverer {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private final ClassLoader classLoader;

    public DuplicateClasspathModuleDiscoverer(ClassLoader classLoader) {
      this.classLoader = classLoader;
    }

    @Override
    public List<MuleModule> discover() {
      List<MuleModule> modules = new LinkedList<>();
      Set<String> moduleNames = new HashSet<>();

      try {
        for (Properties moduleProperties : discoverProperties(classLoader, MODULE_PROPERTIES)) {
          final MuleModule module = createModule(moduleProperties);

          if (moduleNames.contains(module.getName())) {
            logger.warn(format("Module '%s' was already defined", module.getName()));
          }
          moduleNames.add(module.getName());
          modules.add(module);
        }
      } catch (IOException e) {
        throw new RuntimeException("Cannot discover mule modules", e);
      }

      return modules;
    }

    private MuleModule createModule(Properties moduleProperties) {
      // TODO(pablo.kraan): MULE-11796 - remove code duplication
      final String moduleName = (String) moduleProperties.get("module.name");
      Set<String> modulePackages = getExportedPackageByProperty(moduleProperties, EXPORTED_CLASS_PACKAGES_PROPERTY);
      Set<String> modulePaths = getExportedResourcePaths(moduleProperties);
      Set<String> modulePrivilegedPackages =
          getExportedPackageByProperty(moduleProperties, PRIVILEGED_EXPORTED_CLASS_PACKAGES_PROPERTY);
      Set<String> privilegedArtifacts = getPrivilegedArtifactIds(moduleProperties);

      List<ExportedService> exportedServices = getExportedServices(moduleProperties, EXPORTED_SERVICES_PROPERTY);

      return new MuleModule(moduleName, modulePackages, modulePaths, modulePrivilegedPackages, privilegedArtifacts,
                            exportedServices);
    }

    private Set<String> getPrivilegedArtifactIds(Properties moduleProperties) {
      Set<String> privilegedArtifacts;
      final String privilegedArtifactsProperty = (String) moduleProperties.get(PRIVILEGED_ARTIFACTS_PROPERTY);
      Set<String> artifactsIds = new HashSet<>();
      if (!isEmpty(privilegedArtifactsProperty)) {
        for (String artifactName : privilegedArtifactsProperty.split(",")) {
          if (!isEmpty(artifactName.trim())) {
            artifactsIds.add(artifactName);
          }
        }
      }
      privilegedArtifacts = artifactsIds;
      return privilegedArtifacts;
    }

    private Set<String> getExportedPackageByProperty(Properties moduleProperties,
                                                     String privilegedExportedClassPackagesProperty) {
      final String privilegedExportedPackagesProperty = (String) moduleProperties.get(privilegedExportedClassPackagesProperty);
      Set<String> modulePrivilegedPackages;
      if (!isEmpty(privilegedExportedPackagesProperty)) {
        modulePrivilegedPackages = getPackagesFromProperty(privilegedExportedPackagesProperty);
      } else {
        modulePrivilegedPackages = new HashSet<>();
      }
      return modulePrivilegedPackages;
    }

    private Set<String> getExportedResourcePaths(Properties moduleProperties) {
      Set<String> paths = new HashSet<>();
      final String exportedResourcesProperty = (String) moduleProperties.get(EXPORTED_RESOURCE_PROPERTY);
      if (!isEmpty(exportedResourcesProperty)) {
        for (String path : exportedResourcesProperty.split(",")) {
          if (!isEmpty(path.trim())) {
            if (path.startsWith("/")) {
              path = path.substring(1);
            }
            paths.add(path);
          }
        }
      }
      return paths;
    }

    private Set<String> getPackagesFromProperty(String privilegedExportedPackagesProperty) {
      Set<String> packages = new HashSet<>();
      for (String packageName : privilegedExportedPackagesProperty.split(",")) {
        packageName = packageName.trim();
        if (!isEmpty(packageName)) {
          packages.add(packageName);
        }
      }
      return packages;
    }
  }

  private static List<ExportedService> getExportedServices(Properties moduleProperties, String exportedServicesProperty) {
    final String privilegedExportedPackagesProperty = (String) moduleProperties.get(exportedServicesProperty);
    List<ExportedService> exportedServices;
    if (!isEmpty(privilegedExportedPackagesProperty)) {
      exportedServices = getServicesFromProperty(privilegedExportedPackagesProperty);
    } else {
      exportedServices = new ArrayList<>();
    }
    return exportedServices;
  }

  private static List<ExportedService> getServicesFromProperty(String privilegedExportedPackagesProperty) {
    List<ExportedService> exportedServices = new ArrayList<>();

    for (String exportedServiceDefinition : privilegedExportedPackagesProperty.split(",")) {
      String[] split = exportedServiceDefinition.split(":");
      String serviceInterface = split[0];
      String serviceImplementation = split[1];
      URL resource;
      BytesURLStreamHandler bytesURLStreamHandler = new BytesURLStreamHandler(serviceImplementation.getBytes());
      try {
        resource = new URL("custom", "none", 9999, "none", bytesURLStreamHandler);
      } catch (MalformedURLException e) {
        throw new MuleRuntimeException(e);
      }
      exportedServices.add(new ExportedService(serviceInterface, resource));
    }

    return exportedServices;

  }

  /**
   * URL stream handler that read the content from memory
   */
  public static class BytesURLStreamHandler extends URLStreamHandler {

    byte[] content;

    public BytesURLStreamHandler(byte[] content) {
      this.content = content;
    }

    @Override
    public URLConnection openConnection(URL url) {
      return new BytesURLConnection(url, this.content);
    }
  }

  public static class BytesURLConnection extends URLConnection {

    protected byte[] content;

    public BytesURLConnection(URL url, byte[] content) {
      super(url);
      this.content = content;
    }

    @Override
    public void connect() {}

    @Override
    public InputStream getInputStream() {
      return new ByteArrayInputStream(this.content);
    }
  }
}
