/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static java.util.Collections.emptySet;
import static org.codehaus.plexus.util.ExceptionUtils.getStackTrace;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.runtime.extension.api.values.ValueResolvingException.UNKNOWN;
import static org.mule.tooling.client.internal.serialization.SerializationUtilsImpl.deserialize;
import static org.mule.tooling.client.internal.serialization.SerializationUtilsImpl.serialize;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import org.mule.runtime.deployment.model.api.DeploymentException;
import org.mule.tooling.client.api.value.ResolvingFailure;
import org.mule.tooling.client.api.value.Value;
import org.mule.tooling.client.api.value.ValueResult;
import org.mule.tooling.client.api.value.provider.ValueProviderRequest;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.RemoteApplicationInvoker;

import java.util.HashSet;
import java.util.Optional;
import java.util.concurrent.CompletionException;

/**
 * Provides the capability of resolve the {@link Value values} for any Value Provider capable element in a Mule App.
 *
 * @since 1.0
 */
public class DefaultValueProviderService implements ValueProviderService, Command {

  private static final String DEPLOYMENT = "DEPLOYMENT";
  private RemoteApplicationInvoker remoteApplicationInvoker;

  /**
   * Creates an instance of the service.
   *
   * @param remoteApplicationInvoker {@link Application} to deploy and resolve operations against a Mule Runtime. Non null.
   */
  public DefaultValueProviderService(RemoteApplicationInvoker remoteApplicationInvoker) {
    checkNotNull(remoteApplicationInvoker, "remoteApplicationInvoker cannot be null");

    this.remoteApplicationInvoker = remoteApplicationInvoker;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueResult getValues(ValueProviderRequest valueProviderRequest) {
    try {
      return toValueResultDTO(remoteApplicationInvoker.evaluateWithRemoteApplication((applicationId,
                                                                                      runtimeToolingService) -> runtimeToolingService
                                                                                          .getValues(applicationId,
                                                                                                     valueProviderRequest
                                                                                                         .getLocation(),
                                                                                                     valueProviderRequest
                                                                                                         .getProviderName())));
    } catch (Exception e) {
      return toFailureValueResultDTO(e);
    }
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "getValues": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(ValueProviderRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        ValueProviderRequest valueProviderRequest = deserialize(arguments[0]);
        return serialize(getValues(valueProviderRequest));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

  private static ValueResult toValueResultDTO(org.mule.runtime.api.value.ValueResult valueResult) {
    HashSet<Value> values = new HashSet<>();
    for (org.mule.runtime.api.value.Value value : valueResult.getValues()) {
      values.add(toValueDTO(value));
    }

    return new ValueResult(values, toResolvingFailureDTO(valueResult.getFailure()));
  }

  private static Value toValueDTO(org.mule.runtime.api.value.Value value) {
    HashSet<Value> children = new HashSet<>();
    for (org.mule.runtime.api.value.Value child : value.getChilds()) {
      children.add(toValueDTO(child));
    }

    return new Value(value.getId(), value.getDisplayName(), value.getPartName(), children);
  }

  private static ResolvingFailure toResolvingFailureDTO(Optional<org.mule.runtime.api.value.ResolvingFailure> resolvingFailureOptional) {
    if (!resolvingFailureOptional.isPresent()) {
      return null;
    }

    org.mule.runtime.api.value.ResolvingFailure resolvingFailure = resolvingFailureOptional.get();

    return new ResolvingFailure(resolvingFailure.getMessage(), resolvingFailure.getReason(), resolvingFailure.getFailureCode());
  }

  private static ValueResult toFailureValueResultDTO(Throwable t) {

    String failureCode = UNKNOWN;
    if (t instanceof CompletionException) {
      if (t.getCause() instanceof DeploymentException) {
        failureCode = DEPLOYMENT;
      }
    }

    return new ValueResult(emptySet(), new ResolvingFailure(t.getMessage(), getStackTrace(t), failureCode));
  }
}
