/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static org.mule.maven.client.api.MavenClientProvider.discoverProvider;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.MavenClientProvider;
import org.mule.runtime.container.api.ModuleRepository;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginClassLoaderFactory;
import org.mule.runtime.deployment.model.internal.DefaultRegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.internal.RegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.internal.plugin.BundlePluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.plugin.PluginDependenciesResolver;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.DeployableArtifactClassLoaderFactory;
import org.mule.runtime.module.artifact.api.classloader.MuleDeployableArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DescriptorLoaderRepository;
import org.mule.runtime.module.deployment.impl.internal.application.ApplicationDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.application.DeployableMavenClassLoaderModelLoader;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorFactory;
import org.mule.runtime.module.deployment.impl.internal.plugin.ArtifactPluginDescriptorLoader;
import org.mule.runtime.module.deployment.impl.internal.plugin.PluginMavenClassLoaderModelLoader;

import java.util.List;

/**
 * Mule resources registry that has implementations for factories and utilities for creating artifacts.
 */
public class MuleArtifactResourcesRegistry {

  private MavenClient mavenClient;
  private ModuleRepository moduleRepository;

  private ArtifactPluginDescriptorFactory artifactPluginDescriptorFactory;
  private ArtifactPluginDescriptorLoader artifactPluginDescriptorLoader;
  private PluginDependenciesResolver pluginDependenciesResolver;

  private ArtifactClassLoader containerArtifactClassLoader;

  private ApplicationDescriptorFactory applicationDescriptorFactory;

  private RegionPluginClassLoadersFactory regionPluginClassLoadersFactory;

  /**
   * Creates an instance of the registry.
   */
  public MuleArtifactResourcesRegistry(MavenClient mavenClient, ModuleRepository moduleRepository,
                                       ArtifactClassLoader containerArtifactClassLoader) {
    this.mavenClient = mavenClient;
    this.moduleRepository = moduleRepository;
    this.containerArtifactClassLoader = containerArtifactClassLoader;

    init();
  }

  private void init() {
    MavenClientProvider mavenClientProvider = discoverProvider(this.getClass().getClassLoader());

    // Maven Loaders should use current MavenClient configuration
    DescriptorLoaderRepository descriptorLoaderRepository =
        new ToolingDescriptorLoaderRepository(new DeployableMavenClassLoaderModelLoader(mavenClient, mavenClientProvider
            .getLocalRepositorySuppliers()),
                                              new PluginMavenClassLoaderModelLoader(mavenClient, mavenClientProvider
                                                  .getLocalRepositorySuppliers()));
    artifactPluginDescriptorFactory = new ArtifactPluginDescriptorFactory(descriptorLoaderRepository);
    artifactPluginDescriptorLoader = new ArtifactPluginDescriptorLoader(artifactPluginDescriptorFactory);

    applicationDescriptorFactory =
        new ApplicationDescriptorFactory(artifactPluginDescriptorLoader, descriptorLoaderRepository);

    regionPluginClassLoadersFactory = new DefaultRegionPluginClassLoadersFactory(new ArtifactPluginClassLoaderFactory(),
                                                                                 moduleRepository);

    pluginDependenciesResolver = new BundlePluginDependenciesResolver(artifactPluginDescriptorFactory);
  }

  public RegionPluginClassLoadersFactory getRegionPluginClassLoadersFactory() {
    return regionPluginClassLoadersFactory;
  }

  public DeployableArtifactClassLoaderFactory<ArtifactDescriptor> newTemporaryArtifactClassLoaderFactory() {
    return new TemporaryArtifactClassLoaderFactory();
  }

  public ArtifactClassLoader getContainerArtifactClassLoader() {
    return containerArtifactClassLoader;
  }

  public ApplicationDescriptorFactory getApplicationDescriptorFactory() {
    return applicationDescriptorFactory;
  }

  public PluginDependenciesResolver getPluginDependenciesResolver() {
    return pluginDependenciesResolver;
  }

  public ArtifactPluginDescriptorLoader getArtifactPluginDescriptorLoader() {
    return artifactPluginDescriptorLoader;
  }

  /**
   * Creates a class loader instance for a temporary artifact.
   */
  private class TemporaryArtifactClassLoaderFactory
      implements DeployableArtifactClassLoaderFactory<ArtifactDescriptor> {

    /**
     * {@inheritDoc}
     */
    @Override
    public ArtifactClassLoader create(String artifactId, ArtifactClassLoader parent,
                                      ArtifactDescriptor descriptor,
                                      List<ArtifactClassLoader> artifactPluginClassLoaders) {
      return new MuleDeployableArtifactClassLoader(artifactId, descriptor, descriptor.getClassLoaderModel().getUrls(),
                                                   parent.getClassLoader(),
                                                   parent.getClassLoaderLookupPolicy(), artifactPluginClassLoaders);
    }

  }


}
