/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;

import java.io.File;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Provides access to Mule Runtime {@link ExtensionModel}.
 *
 * @since 1.0
 */
public interface MuleRuntimeExtensionModelProvider {

  /**
   * Gets the {@link ExtensionModel} for the given plugin.
   *
   * @param pluginDescriptor {@link ArtifactDescriptor} referencing to the plugin artifact.
   * @return {@link ExtensionModel} for the plugin or {@link Optional#empty()} if the plugin doesn't have an
   *         {@link ExtensionModel}.
   */
  Optional<ExtensionModel> getExtensionModel(ArtifactDescriptor pluginDescriptor);

  /**
   * Gets the min Mule Runtime version for a given plugin.
  
   * @param pluginDescriptor {@link ArtifactDescriptor} referencing to the plugin artifact.
   * @return the min Mule Runtime version.
   */
  Optional<String> getMinMuleVersion(ArtifactDescriptor pluginDescriptor);

  /**
   * Gets the {@link ExtensionModel} for the given plugin.
   *
   * @param plugin {@link File} referencing to the plugin jar file or the unzipped folder.
   * @return {@link ExtensionModel} for the plugin or {@link Optional#empty()} if the plugin doesn't have an
   *         {@link ExtensionModel}.
   */
  Optional<ExtensionModel> getExtensionModel(File plugin);

  /**
   * Gets the min Mule Runtime version for a given plugin.
  
   * @param plugin {@link File} referencing to the plugin artifact.
   * @return the min Mule Runtime version.
   */
  Optional<String> getMinMuleVersion(File plugin);

  /**
   * Gets the schema for the given plugin.
   *
   * @param plugin {@link File} referencing to the plugin jar file or the unzipped folder.
   * @return {@link String} with the schema content or {@link Optional#empty()} if the plugin doesn't have an
   *         {@link ExtensionModel}.
   */
  Optional<String> getExtensionSchema(File plugin);

  /**
   * Gets the schema for the given plugin.
   *
   * @param pluginDescriptor {@link ArtifactDescriptor} referencing to the plugin artifact.
   * @return {@link String} with the schema content or {@link Optional#empty()} if the plugin doesn't have an
   *         {@link ExtensionModel}.
   */
  Optional<String> getExtensionSchema(ArtifactDescriptor pluginDescriptor);

  /**
   * Gets the {@link ExtensionModel}s for the Mule Runtime core components.
   *
   * @return a {@link List} with the {@link ExtensionModel}s for Mule Runtime core components.
   */
  List<ExtensionModel> getRuntimeExtensionModels();
}
