/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.util.Objects.requireNonNull;
import org.mule.tooling.agent.RuntimeToolingService;

import java.net.URL;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Base class implementation for {@link ApplicationDeployer}.
 *
 * @since 4.0
 */
public abstract class AbstractApplicationDeployer implements ApplicationDeployer {

  protected final Logger LOGGER = LoggerFactory.getLogger(this.getClass());

  private URL applicationUrlContent;
  private RuntimeToolingService runtimeToolingService;

  /**
   * Creates an instance of the context.
   *
   * @param applicationUrlContent {@link URL} to the application content.
   * @param runtimeToolingService {@link RuntimeToolingService} for resolving operations against the Mule Runtime.
   */
  public AbstractApplicationDeployer(URL applicationUrlContent,
                                     RuntimeToolingService runtimeToolingService) {
    requireNonNull(applicationUrlContent, "applicationUrlContent cannot be null");
    requireNonNull(runtimeToolingService, "runtimeToolingService cannot be null");

    this.applicationUrlContent = applicationUrlContent;
    this.runtimeToolingService = runtimeToolingService;
  }

  @Override
  public String deploy() {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Deploying application to Mule Runtime");
    }
    String applicationId = doInternalInit(applicationUrlContent, runtimeToolingService);
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Application deployed with applicationId: {}", applicationId);
    }
    return applicationId;
  }

  protected abstract String doInternalInit(URL applicationUrlContent, RuntimeToolingService runtimeToolingService);

}
