/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static com.google.common.io.Files.createTempDir;
import static java.lang.String.format;
import static java.nio.file.Files.createTempDirectory;
import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.io.FileUtils.deleteQuietly;
import static org.apache.commons.io.FileUtils.toFile;
import static org.apache.commons.io.IOUtils.write;
import static org.mule.runtime.core.api.util.FileUtils.createTempFile;
import static org.mule.runtime.core.api.util.FileUtils.unzip;
import static org.mule.tooling.client.internal.application.RemoteApplicationContextFactory.createRemoteApplicationContext;
import static org.mule.tooling.client.internal.utils.IOUtils.readContentFromUrl;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.DefaultApplicationModelFactory;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;

/**
 * Represents an application and allows to be deployed into Mule Runtime.
 * <p/>
 * This class is not thread-safe, therefore if there is the need to support it this would be the place to change in order
 * to allow blocking operations to Mule Runtime.
 *
 * @since 4.0
 */
public class DefaultApplication implements Application {

  private static final int CONNECT_TIMEOUT = 5000;
  private static final int READ_TIMEOUT = 5000;

  private static final Logger LOGGER = getLogger(DefaultApplication.class);

  private static final String FILE_PROTOCOL = "file";

  private URL applicationUrlContent;
  private Map<String, String> properties;
  private ToolingArtifactContext context;

  private LazyValue<List<ExtensionModel>> extensionModels;
  private LazyValue<ToolingArtifactClassLoader> applicationClassLoader;
  private LazyValue<ApplicationModel> applicationModel;

  private LazyValue<ApplicationResources> applicationResources;
  private String remoteApplicationId;

  private AtomicBoolean disposed = new AtomicBoolean(false);

  public DefaultApplication(URL applicationUrlContent, ToolingArtifactContext context, Map<String, String> properties) {
    requireNonNull(applicationUrlContent, "applicationUrlContent cannot be null");
    requireNonNull(context, "context cannot be null");
    requireNonNull(properties, "properties cannot be null");

    this.applicationUrlContent = applicationUrlContent;
    this.context = context;
    this.properties = properties;

    this.applicationResources = newApplicationResourcesLazyValue();
    this.applicationModel = newApplicationModelLazyValue();
    this.applicationClassLoader = newApplicationClassLoaderLazyValue();
    this.extensionModels = newExtensionModelsLazyValue();
  }

  @Override
  public synchronized void setContext(ToolingArtifactContext context) {
    checkState();
    if (this.context != null && this.context.getAgentConfiguration().isPresent() && context.getAgentConfiguration().isPresent()) {
      if (!this.context.getAgentConfiguration().get().getToolingApiUrl()
          .equals(context.getAgentConfiguration().get().getToolingApiUrl())) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Application context has changed, trying to dispose (if present) the remote application");
        }
        disposeRemoteApplicationContext();
      }
    }
    this.context = context;
  }

  @Override
  public List<ExtensionModel> getExtensionModels() {
    checkState();
    return extensionModels.get();
  }

  private boolean isFileProtocol() {
    return applicationUrlContent.getProtocol().equals(FILE_PROTOCOL);
  }

  @Override
  public String getApplicationName() {
    checkState();
    return getArtifactClassLoader().getArtifactId();
  }

  @Override
  public Map<String, String> getProperties() {
    return properties;
  }

  @Override
  public URL getApplicationUrlContent() {
    checkState();
    return applicationUrlContent;
  }

  @Override
  public ArtifactClassLoader getArtifactClassLoader() {
    checkState();
    return applicationClassLoader.get();
  }

  private ApplicationDescriptor getApplicationDescriptor() {
    checkState();
    return (ApplicationDescriptor) getArtifactClassLoader().getArtifactDescriptor();
  }

  @Override
  public synchronized <R> R evaluateWithRemoteApplication(ApplicationRemoteFunction<R> function) {
    checkState();
    RuntimeToolingService runtimeToolingService = context.getRuntimeToolingService();
    try {
      if (remoteApplicationId == null) {
        remoteApplicationId = createRemoteApplicationContext(this.applicationUrlContent, runtimeToolingService).deploy();
      }
      return doInvoke(function, remoteApplicationId, runtimeToolingService);
    } catch (NoSuchApplicationException e) {
      disposeRemoteApplicationQuietly(runtimeToolingService);
      remoteApplicationId = createRemoteApplicationContext(this.applicationUrlContent, runtimeToolingService).deploy();
      return doInvoke(function, remoteApplicationId, runtimeToolingService);
    }
  }

  @Override
  public synchronized void dispose() {
    checkState();
    if (!disposed.getAndSet(true)) {
      disposeClassLoader();
      disposeResources();
      disposeRemoteApplicationContext();
    }
  }

  private void disposeRemoteApplicationContext() {
    if (remoteApplicationId != null) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Disposing Mule Runtime remote application for applicationId: {}", remoteApplicationId);
      }
      disposeRemoteApplicationQuietly(context.getRuntimeToolingService());
      remoteApplicationId = null;
    }
  }

  private void disposeResources() {
    if (applicationResources.isComputed()) {
      final ApplicationResources applicationResources = this.applicationResources.get();
      if (!isFileProtocol()) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Deleting temporary file used for application");
        }
        File applicationRootFile = applicationResources.getApplicationRootFile();
        boolean fileDeleted = deleteQuietly(applicationRootFile);
        if (!fileDeleted) {
          LOGGER.warn("Couldn't delete temporary application file: {}", applicationRootFile.getAbsoluteFile());
        }
      }
      File workingDirectory = applicationResources.getWorkingDirectory();
      if (workingDirectory != null && workingDirectory.exists()) {
        boolean fileDeleted = deleteQuietly(workingDirectory);
        if (!fileDeleted) {
          LOGGER.warn("Couldn't delete temporary working application file: {}", workingDirectory.getAbsoluteFile());
        }
      }
      this.applicationResources = null;
      this.applicationModel = null;
      this.extensionModels = null;
    }
  }

  private void disposeClassLoader() {
    if (applicationClassLoader != null && applicationClassLoader.isComputed()) {
      try {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Disposing application class loader");
        }
        applicationClassLoader.get().dispose();
      } finally {
        applicationClassLoader = null;
      }
    }
  }

  private void disposeRemoteApplicationQuietly(RuntimeToolingService runtimeToolingService) {
    try {
      runtimeToolingService.disposeApplication(remoteApplicationId);
    } catch (Exception e) {
      // Nothing to do here...
    }
  }

  private <R> R doInvoke(ApplicationRemoteFunction<R> function, String remoteApplicationId,
                         RuntimeToolingService runtimeToolingService) {
    return function.apply(remoteApplicationId, runtimeToolingService);
  }

  private boolean isApplicationExploded() {
    return isFileProtocol() && toFile(applicationUrlContent).isDirectory();
  }

  @Override
  public ApplicationModel getApplicationModel() {
    checkState();
    return applicationModel.get();
  }

  private LazyValue<ToolingArtifactClassLoader> newApplicationClassLoaderLazyValue() {
    return new LazyValue<>(() -> {
      final ApplicationResources applicationResources = this.applicationResources.get();
      final File applicationRootFile = applicationResources.getApplicationRootFile();
      try {
        return context.getApplicationService().createApplicationClassLoader(applicationRootFile.getName(), applicationRootFile,
                                                                            applicationResources.getWorkingDirectory());
      } catch (IOException e) {
        throw new ToolingException(
                                   format("Error while creating application class loader for application URL: %s using temporary application file: %s",
                                          applicationUrlContent, applicationRootFile));
      }
    });
  }

  private LazyValue<ApplicationModel> newApplicationModelLazyValue() {
    return new LazyValue<>(() -> new DefaultApplicationModelFactory()
        .createApplicationModel(getApplicationDescriptor(),
                                ImmutableSet.<ExtensionModel>builder().addAll(getExtensionModels()).build(),
                                getArtifactClassLoader().getClassLoader())
        .orElseThrow(() -> new ToolingException(format("Couldn't create ApplicationModel from %s", this))));
  }

  private LazyValue<List<ExtensionModel>> newExtensionModelsLazyValue() {
    return new LazyValue<>(() -> {
      final List<ExtensionModel> extensionModels = applicationClassLoader.get().getArtifactPluginClassLoaders().stream()
          .map(artifactPluginClassLoader -> artifactPluginClassLoader.getArtifactDescriptor().getBundleDescriptor())
          .map(bundleDescriptor -> context.getMuleRuntimeExtensionModelProvider()
              .getExtensionModel(toArtifactDescriptor(bundleDescriptor)).orElse(null))
          .filter(extensionModel -> extensionModel != null)
          .collect(toList());
      extensionModels.addAll(context.getMuleRuntimeExtensionModelProvider().getRuntimeExtensionModels());
      return extensionModels;
    });
  }

  private org.mule.tooling.client.api.descriptors.ArtifactDescriptor toArtifactDescriptor(BundleDescriptor bundleDescriptor) {
    final org.mule.tooling.client.api.descriptors.ArtifactDescriptor.Builder builder =
        org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder()
            .withGroupId(bundleDescriptor.getGroupId())
            .withArtifactId(bundleDescriptor.getArtifactId())
            .withVersion(bundleDescriptor.getVersion())
            .withExtension(bundleDescriptor.getType());
    bundleDescriptor.getClassifier().ifPresent(classifier -> builder.withClassifier(classifier));
    return builder.build();
  }

  private LazyValue<ApplicationResources> newApplicationResourcesLazyValue() {
    return new LazyValue<>(() -> {
      File tempFile = null;
      try {
        if (isApplicationExploded()) {
          if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Application URL already references to a file location with the content exploded");
          }
          return new ApplicationResources(toFile(applicationUrlContent), createTempDir());
        } else {
          if (LOGGER.isDebugEnabled()) {
            LOGGER
                .debug(
                       "Application URL either references to a remote location or a file location with packaged jar, copying to a temporary folder");
          }
          File applicationRootFile = createTempDirectory("application").toFile();
          // Just in case if an error happens and it is not disposed the application
          applicationRootFile.deleteOnExit();

          if (!isFileProtocol()) {
            // TODO this should be configured in Builder for tooling runtimne client (TLS may be needed too)
            byte[] fileContent = readContentFromUrl(applicationUrlContent, CONNECT_TIMEOUT, READ_TIMEOUT);
            tempFile = createTempFile("application", ".jar");
            try (FileOutputStream outputStream = new FileOutputStream(tempFile)) {
              write(fileContent, outputStream);
            }
          } else {
            tempFile = toFile(applicationUrlContent);
          }
          unzip(tempFile, applicationRootFile);
          return new ApplicationResources(applicationRootFile, applicationRootFile);
        }
      } catch (IOException e) {
        throw new ToolingException(format("Couldn't expand the application or content to a temporary folder from URL: %s",
                                          applicationUrlContent),
                                   e);
      } finally {
        if (tempFile != null && !isFileProtocol()) {
          boolean fileDeleted = deleteQuietly(tempFile);
          if (!fileDeleted) {
            LOGGER.warn("Couldn't delete temporary application file: {}", tempFile.getAbsoluteFile());
          }
        }
      }
    });
  }

  @Override
  public String toString() {
    return this.getClass().getSimpleName() + "{applicationUrlContent=" + applicationUrlContent
        + (applicationResources != null && applicationResources.isComputed()
            ? ", applicationRootFile=" + applicationResources.get().getApplicationRootFile().getAbsolutePath() + "}" : "}");
  }

  private void checkState() {
    Preconditions.checkState(!disposed.get(), "Application already disposed, cannot be used anymore");
  }

  private class ApplicationResources {

    private File applicationRootFile;
    private File workingDirectory;

    public ApplicationResources(File applicationRootFile, File workingDirectory) {
      this.applicationRootFile = applicationRootFile;
      this.workingDirectory = workingDirectory;
    }

    public File getApplicationRootFile() {
      return applicationRootFile;
    }

    public File getWorkingDirectory() {
      return workingDirectory;
    }

  }

}
