/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import static java.lang.String.format;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.tooling.client.api.el.Severity.ERROR;
import static org.mule.tooling.client.api.el.Severity.WARNING;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.serialization.SerializationUtilsImpl.deserialize;
import static org.mule.tooling.client.internal.serialization.SerializationUtilsImpl.serialize;
import static org.mule.weave.v2.module.pojo.JavaDataFormat.isJavaMimeType;
import org.mule.datasense.api.metadataprovider.ToolingHelper;
import org.mule.metadata.message.api.el.ExpressionLanguageMetadataTypeResolver;
import org.mule.tooling.client.api.dataweave.DataWeaveModuleOption;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.dataweave.validation.DataWeaveValidationRequest;
import org.mule.tooling.client.api.el.Location;
import org.mule.tooling.client.api.el.Position;
import org.mule.tooling.client.api.el.ValidationMessage;
import org.mule.tooling.client.api.el.ValidationResult;
import org.mule.tooling.client.internal.Command;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.weave.v2.module.DataFormat;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

public class DefaultDataWeaveService implements DataWeaveService, Command {

  private final Application application;
  private DataWeaveRunnerProvider runnerProvider;
  private DataWeaveRunner dataWeaveRunner;
  private ModulesAnalyzer modulesAnalyzer;

  public DefaultDataWeaveService(Application application, DataWeaveRunnerProvider runnerProvider,
                                 ModulesAnalyzer modulesAnalyzer) {
    this.application = application;
    this.runnerProvider = runnerProvider;
    this.modulesAnalyzer = modulesAnalyzer;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    final EventModel event = parameters.getEvent();
    if (event != null && anyInputIsJavaBased(event)) {
      throw new IllegalArgumentException("Java input not supported, serialize to DW script");
    }
    return (dataWeaveRunner == null ? getRunnerProvider().getRunner(parameters) : dataWeaveRunner).execute(parameters);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValidationResult validate(DataWeaveValidationRequest request) {
    return withContextClassLoader(application.getArtifactClassLoader().getClassLoader(), () -> {
      List<ValidationMessage> validationMessages = new ArrayList<>();
      ExpressionLanguageMetadataTypeResolver.getInstance().getOutputType(
                                                                         ToolingHelper
                                                                             .resolveTypeBindings(request.getEventType(),
                                                                                                  request.getGlobalBindings(),
                                                                                                  request.getFunctionBindings()),
                                                                         request.getScript(),
                                                                         new ExpressionLanguageMetadataTypeResolver.MessageCallback() {

                                                                           @Override
                                                                           public void warning(String message,
                                                                                               ExpressionLanguageMetadataTypeResolver.MessageLocation location) {
                                                                             validationMessages
                                                                                 .add(new ValidationMessage(WARNING, message,
                                                                                                            toLocation(location)));

                                                                           }

                                                                           @Override
                                                                           public void error(String message,
                                                                                             ExpressionLanguageMetadataTypeResolver.MessageLocation location) {
                                                                             validationMessages
                                                                                 .add(new ValidationMessage(ERROR, message,
                                                                                                            toLocation(location)));
                                                                           }

                                                                         });
      return new ValidationResult(validationMessages);
    });
  }

  private Location toLocation(ExpressionLanguageMetadataTypeResolver.MessageLocation location) {
    final ExpressionLanguageMetadataTypeResolver.MessagePosition startPosition = location.getStartPosition();
    final ExpressionLanguageMetadataTypeResolver.MessagePosition endPosition = location.getEndPosition();
    return new Location(new Position(startPosition.getLine(), startPosition.getColumn(), startPosition.getOffset()),
                        new Position(endPosition.getLine(), endPosition.getColumn(), endPosition.getOffset()));
  }

  private boolean anyInputIsJavaBased(final EventModel event) {
    final MessageModel message = event.getMessage();
    if (isJavaBased(message.getPayload()) || isJavaBased(message.getAttributes())) {
      return true;
    }

    final Map<String, TypedValueModel> variables = event.getVariables();
    if (variables != null && variables.values().stream().anyMatch(this::isJavaBased)) {
      return true;
    }

    // otherwise
    return false;
  }

  private boolean isJavaBased(final TypedValueModel model) {
    return model != null && isJavaMimeType(model.getDataType().getMediaType());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<DataWeaveModuleOption> getReaderOptions(String contentType) {
    return getModulesAnalyzer().getOptions(contentType, DataFormat::readerOptions);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<DataWeaveModuleOption> getWriterOptions(String contentType) {
    return getModulesAnalyzer().getOptions(contentType, DataFormat::writerOptions);
  }

  public ModulesAnalyzer getModulesAnalyzer() {
    return modulesAnalyzer;
  }

  public DataWeaveRunnerProvider getRunnerProvider() {
    return runnerProvider;
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "execute": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(DataWeavePreviewRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serialize(execute(deserialize(arguments[0])));
      }
      case "validate": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(DataWeaveValidationRequest.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serialize(validate(deserialize(arguments[0])));
      }
      case "getReaderOptions": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serialize(getReaderOptions(deserialize(arguments[0])));
      }
      case "getWriterOptions": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serialize(getWriterOptions(deserialize(arguments[0])));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
