/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.serialization.XStreamServerSerializer.serialize;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingService;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.dataweave.DataWeaveService;
import org.mule.tooling.client.api.metadata.MetadataService;
import org.mule.tooling.client.api.value.provider.ValueProviderService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.dataweave.DataWeaveRunner;
import org.mule.tooling.client.internal.dataweave.DataWeaveRunnerProvider;
import org.mule.tooling.client.internal.dataweave.DefaultDataWeaveService;
import org.mule.tooling.client.internal.dataweave.LocalRunner;
import org.mule.tooling.client.internal.dataweave.ModulesAnalyzer;
import org.mule.tooling.client.internal.dataweave.RemoteRunner;

import java.util.Map;

/**
 * Implementation that allows to do Tooling operations without deploying the application until it is needed.
 * <p/>
 * If an operation requires the application to be deployed it will deploy it first and then delegate the operation to the default
 * implementation of {@link ToolingArtifact}.
 *
 * @since 4.0
 */
public class DefaultToolingArtifact implements ToolingArtifact, Command {

  private String id;
  private Application application;

  private ConnectivityTestingService connectivityTestingService;
  private MetadataService metadataService;
  private DataSenseService dataSenseService;
  private DataWeaveService dataWeaveService;
  private ValueProviderService valueProviderService;

  /**
   * Creates an instance of the {@link DefaultToolingArtifact} from a fetched applicationId or deploys the application to obtain
   * an identifier in case if null.
   *
   * @param id {@link String} identifier for this {@link ToolingArtifact}. Non null.
   * @param metadataCache {@link MetadataCache} to be used by this {@link ToolingArtifact}. Non null.
   * @param application {@link Application} to handle state, local and remote (Mule Runtime) in order to resolve operations. Non null.
   */
  public DefaultToolingArtifact(String id, MetadataCache metadataCache, Application application) {
    checkNotNull(id, "id cannot be null");
    checkNotNull(application, "application cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");

    this.id = id;
    this.application = application;

    // Services registration...
    this.connectivityTestingService = new DefaultConnectivityTestingService(application);
    MetadataProvider metadataProvider = new InternalMetadataProvider(application);
    this.metadataService = new ToolingMetadataServiceAdapter(application, metadataProvider, metadataCache);
    this.dataSenseService = new DefaultDataSenseService(application, metadataProvider, metadataCache);

    DataWeaveRunner remoteRunner = new RemoteRunner(application);
    DataWeaveRunner localRunner = new LocalRunner(application);
    this.dataWeaveService =
        new DefaultDataWeaveService(application, new DataWeaveRunnerProvider(localRunner, remoteRunner),
                                    new ModulesAnalyzer());
    this.valueProviderService = new DefaultValueProviderService(application);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public String getId() {
    return id;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Map<String, String> getProperties() {
    return application.getProperties();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConnectivityTestingService connectivityTestingService() {
    return this.connectivityTestingService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataService metadataService() {
    return this.metadataService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataSenseService dataSenseService() {
    return this.dataSenseService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeaveService dataWeaveService() {
    return dataWeaveService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ValueProviderService valueProviderService() {
    return valueProviderService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void dispose() {
    application.dispose();
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "id": {
        return serialize(this.getId());
      }
      case "connectivityTestingService": {
        return this.connectivityTestingService();
      }
      case "metadataService": {
        return this.metadataService();
      }
      case "dataSenseService": {
        return this.dataSenseService();
      }
      case "dataWeaveService": {
        return this.dataWeaveService();
      }
      case "valueProviderService": {
        return this.valueProviderService();
      }
      case "dispose": {
        this.dispose();
        return null;
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }
}
