
/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.empty;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.application.MuleApplicationClassLoaderFactory;
import org.mule.runtime.deployment.model.internal.nativelib.ArtifactCopyNativeLibraryFinder;
import org.mule.runtime.deployment.model.internal.nativelib.NativeLibraryFinder;
import org.mule.runtime.deployment.model.internal.nativelib.NativeLibraryFinderFactory;
import org.mule.runtime.deployment.model.internal.tooling.ToolingApplicationClassLoaderBuilder;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.MuleArtifactResourcesRegistry;

import com.google.common.collect.ImmutableList;

import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.util.List;

/**
 * Default implementation of {@link ApplicationService}.
 *
 * @since 4.0
 */
public class DefaultApplicationService implements ApplicationService {

  private MuleArtifactResourcesRegistry muleArtifactResourcesRegistry;

  public DefaultApplicationService(MuleArtifactResourcesRegistry muleArtifactResourcesRegistry) {
    requireNonNull(muleArtifactResourcesRegistry, "muleArtifactResourcesRegistry cannot be null");

    this.muleArtifactResourcesRegistry = muleArtifactResourcesRegistry;
  }

  @Override
  public ToolingArtifactClassLoader createApplicationClassLoader(String applicationName, File applicationLocation,
                                                                 File workingDirectory)
      throws IOException {
    try {
      ToolingApplicationClassLoaderBuilder builder =
          new ToolingApplicationClassLoaderBuilder(new MuleApplicationClassLoaderFactory(new ToolingNativeLibraryFinderFactory(workingDirectory)),
                                                   muleArtifactResourcesRegistry.getRegionPluginClassLoadersFactory());
      builder.setParentClassLoader(muleArtifactResourcesRegistry.getContainerArtifactClassLoader());

      ApplicationDescriptor artifactDescriptor =
          muleArtifactResourcesRegistry.getApplicationDescriptorFactory().create(applicationLocation, empty());
      builder.setArtifactDescriptor(artifactDescriptor);

      List<ArtifactPluginDescriptor> resolvedArtifactPluginDescriptors =
          muleArtifactResourcesRegistry.getPluginDependenciesResolver()
              .resolve(ImmutableList.<ArtifactPluginDescriptor>builder().addAll(artifactDescriptor.getPlugins()).build());
      resolvedArtifactPluginDescriptors.stream().forEach(builder::addArtifactPluginDescriptors);
      return builder.build();
    } catch (Exception e) {
      throw new ToolingException("Error while creating application class loader", e);
    }
  }

  private class ToolingNativeLibraryFinderFactory implements NativeLibraryFinderFactory {

    private File workingDirectory;

    public ToolingNativeLibraryFinderFactory(File workingDirectory) {
      this.workingDirectory = workingDirectory;
    }

    @Override
    public NativeLibraryFinder create(String name, URL[] urls) {
      return new ArtifactCopyNativeLibraryFinder(new File(new File(workingDirectory, name), "temp"), urls);
    }
  }
}
