/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dsl;

import static java.lang.String.format;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.Command.notMatchingMethod;
import static org.mule.tooling.client.internal.serialization.XStreamServerSerializer.deserialize;
import static org.mule.tooling.client.internal.serialization.XStreamServerSerializer.serialize;
import org.mule.metadata.api.model.MetadataType;
import org.mule.tooling.client.api.artifact.dsl.DslElementSyntax;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolver;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.config.ConfigurationModel;
import org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.operation.OperationModel;
import org.mule.tooling.client.api.extension.model.source.SourceModel;
import org.mule.tooling.client.internal.Command;

import java.util.List;
import java.util.Optional;

/**
 * Dispatcher for the requests incoming from tooling-api to the actual
 * @since 1.0
 */
class ServiceDelegatingDslSyntaxResolver implements DslSyntaxResolver, Command {

  private final DefaultDslSyntaxResolverService delegate;
  private final ExtensionModel toolingModel;
  private final List<ArtifactDescriptor> pluginArtifactDescriptors;

  ServiceDelegatingDslSyntaxResolver(DefaultDslSyntaxResolverService service,
                                     ExtensionModel toolingModel,
                                     List<ArtifactDescriptor> pluginArtifactDescriptors) {
    this.delegate = service;
    this.toolingModel = toolingModel;
    this.pluginArtifactDescriptors = pluginArtifactDescriptors;
  }

  @Override
  public DslElementSyntax resolve(ConfigurationModel component) {
    return delegate.resolve(component, toolingModel, pluginArtifactDescriptors);
  }

  @Override
  public DslElementSyntax resolve(ConnectionProviderModel component) {
    return delegate.resolve(component, toolingModel, pluginArtifactDescriptors);
  }

  @Override
  public DslElementSyntax resolve(ConstructModel component) {
    return delegate.resolve(component, toolingModel, pluginArtifactDescriptors);
  }

  @Override
  public DslElementSyntax resolve(OperationModel component) {
    return delegate.resolve(component, toolingModel, pluginArtifactDescriptors);
  }

  @Override
  public DslElementSyntax resolve(SourceModel component) {
    return delegate.resolve(component, toolingModel, pluginArtifactDescriptors);
  }

  @Override
  public Optional<DslElementSyntax> resolve(MetadataType type) {
    return delegate.resolve(type, toolingModel, pluginArtifactDescriptors);
  }

  @Override
  public void dispose() {
    delegate.invalidate(pluginArtifactDescriptors, toolingModel);
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "resolve": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        if (classes[0].equals(ConfigurationModel.class.getName())) {
          return serialize(resolve((ConfigurationModel) deserialize(arguments[0])));
        }
        if (classes[0].equals(ConnectionProviderModel.class.getName())) {
          return serialize(resolve((ConnectionProviderModel) deserialize(arguments[0])));
        }
        if (classes[0].equals(ConstructModel.class.getName())) {
          return serialize(resolve((ConstructModel) deserialize(arguments[0])));
        }
        if (classes[0].equals(OperationModel.class.getName())) {
          return serialize(resolve((OperationModel) deserialize(arguments[0])));
        }
        if (classes[0].equals(SourceModel.class.getName())) {
          return serialize(resolve((SourceModel) deserialize(arguments[0])));
        }
        if (classes[0].equals(MetadataType.class.getName())) {
          return serialize(resolve((MetadataType) deserialize(arguments[0])));
        }
        throw notMatchingMethod(this.getClass(), methodName, classes);
      }
      case "dispose": {
        dispose();
        return null;
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
