/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.serialization;

import static java.util.Collections.synchronizedList;

import com.thoughtworks.xstream.XStream;
import com.thoughtworks.xstream.converters.MarshallingContext;
import com.thoughtworks.xstream.converters.UnmarshallingContext;
import com.thoughtworks.xstream.converters.collections.MapConverter;
import com.thoughtworks.xstream.io.HierarchicalStreamReader;
import com.thoughtworks.xstream.io.HierarchicalStreamWriter;
import com.thoughtworks.xstream.mapper.Mapper;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * Serialization support that allows to serialize/deserialze objects from this implementation.
 *
 * @since 4.0
 */
public final class XStreamServerSerializer {

  private static XStream createXStream() {
    final XStream xStream = XStreamFactory.createXStream();
    xStream
        .registerConverter(new TypeAnnotationsMapConverter(XStreamClientSerializer.class.getClassLoader().getParent().getParent(),
                                                           synchronizedList(new ArrayList<>()),
                                                           xStream.getMapper()));
    xStream.registerConverter(new XmlArtifactDeserializationRequestConverter(xStream.getMapper()));
    xStream.ignoreUnknownElements();
    return xStream;
  }

  private static class TypeAnnotationsMapConverter extends MapConverter {

    private TypeAnnotationMapWriter typeAnnotationMapWriter;
    private TypeAnnotationMapReader typeAnnotationMapReader;

    public TypeAnnotationsMapConverter(ClassLoader targetClassLoader,
                                       List<Class> typeAnnotationsNotSupportedByToolingRuntimeClient, Mapper mapper) {
      super(mapper);
      this.typeAnnotationMapWriter =
          new TypeAnnotationMapWriter(targetClassLoader, typeAnnotationsNotSupportedByToolingRuntimeClient);
      this.typeAnnotationMapReader = new TypeAnnotationMapReader();
    }

    @Override
    public void marshal(Object source, HierarchicalStreamWriter writer, MarshallingContext context) {
      super.marshal(typeAnnotationMapWriter.write((Map) source), writer, context);
    }

    @Override
    public Object unmarshal(HierarchicalStreamReader reader, UnmarshallingContext context) {
      Map output = (Map) super.unmarshal(reader, context);
      return typeAnnotationMapReader.read(output);
    }
  }

  public static String serialize(Object object) {
    return createXStream().toXML(object);
  }

  public static <T> T deserialize(String content) {
    return (T) createXStream().fromXML(content);
  }

}
