/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.beust.jcommander.internal.Lists.newArrayList;
import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.client.WireMock.urlEqualTo;
import static com.github.tomakehurst.wiremock.core.WireMockConfiguration.wireMockConfig;
import static java.util.Collections.emptyMap;
import static java.util.Optional.of;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.collection.IsCollectionWithSize.hasSize;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyObject;
import static org.mockito.Matchers.anyString;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;
import static org.mockito.Mockito.when;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;
import org.mule.maven.client.api.MavenClient;
import org.mule.maven.client.api.model.BundleDependency;
import org.mule.maven.client.api.model.BundleDescriptor;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactDescriptor;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.internal.application.ApplicationService;
import org.mule.tooling.client.internal.application.DefaultApplication;

import com.github.tomakehurst.wiremock.junit.WireMockClassRule;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@Feature("DataSense Session")
@Story("DefaultApplication provides the list of extension models, class loader, application model and remote application context")
@RunWith(MockitoJUnitRunner.class)
public class DefaultApplicationTestCase {

  private static final String APP_ID = "appId";
  private static final String APPLICATIONS_DATASENSE_STATIC = "applications/datasense-static";
  private static final String JAR_EXTENSION = ".jar";
  @Mock
  private RuntimeToolingService runtimeToolingService;
  @Mock
  private ToolingArtifactContext context;
  @Mock
  private MavenClient mavenClient;
  @Mock
  private MuleRuntimeExtensionModelProvider runtimeExtensionModelProvider;
  @Mock
  private ToolingArtifactClassLoader applicationClassLoader;
  @Mock
  private ApplicationDescriptor applicationDescriptor;
  @Mock
  private ExtensionModel runtimeExtensionModel;
  @Mock
  private ExtensionModel extensionModel;
  @Mock
  private ApplicationService applicationService;
  @Rule
  public WireMockClassRule wireMockRule = new WireMockClassRule(wireMockConfig().dynamicPort());
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public ExpectedException expectedException = none();

  private File artifactFile;
  private File artifactJarFile;
  private URL applicationContentUrl;

  private final String REQUEST_PATH = "/app";

  @Before
  public void before() throws Exception {
    artifactFile = toFile(this.getClass().getClassLoader().getResource(APPLICATIONS_DATASENSE_STATIC));
    artifactJarFile = toFile(this.getClass().getClassLoader().getResource(APPLICATIONS_DATASENSE_STATIC + JAR_EXTENSION));
    applicationContentUrl = artifactFile.toURI().toURL();

    when(context.getRuntimeToolingService()).thenReturn(runtimeToolingService);
  }

  @Test
  @Description("Validates that when application is created from a remote content it will call the deployment with the inputStream")
  public void deploymentRemoteApplicationUrlContentShouldBeByInputStream() throws Exception {
    File jarFile = temporaryFolder.newFile();
    compress(jarFile, artifactFile);

    wireMockRule.stubFor(
                         get(urlEqualTo("/app"))
                             .willReturn(aResponse()
                                 .withStatus(200)
                                 .withHeader("Content-Type", "application/octet-stream")
                                 .withBody(IOUtils.toByteArray(jarFile.toURI().toURL()))));
    int port = wireMockRule.port();

    applicationContentUrl = new URI("http://localhost:" + port + REQUEST_PATH).toURL();
    when(runtimeToolingService.deployApplication(any(InputStream.class))).thenReturn(APP_ID);

    DefaultApplication application = new DefaultApplication(applicationContentUrl, context, emptyMap());
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(APP_ID));

    verify(runtimeToolingService).deployApplication(any(InputStream.class));
  }

  @Test
  @Description("Validates that when application is created from a local content it will call the deployment with the path")
  public void deploymentLocalApplicationUrlContentShouldBeByPath() throws Exception {
    when(runtimeToolingService.deployApplication(artifactFile)).thenReturn(APP_ID);

    DefaultApplication application =
        new DefaultApplication(applicationContentUrl, context, emptyMap());
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(APP_ID));

    verify(runtimeToolingService).deployApplication(artifactFile);
  }

  @Test
  @Description("Validates that if a remote application was deployed it will be disposed when application is disposed")
  public void disposeRemoteApplicationIfDeployed() throws Exception {
    when(runtimeToolingService.deployApplication(artifactFile)).thenReturn(APP_ID);

    DefaultApplication application =
        new DefaultApplication(applicationContentUrl, context, emptyMap());
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(APP_ID));
    application.dispose();

    verify(runtimeToolingService).deployApplication(artifactFile);
    verify(runtimeToolingService).disposeApplication(APP_ID);
  }

  @Test
  @Description("Validates that if a remote application was not deployed when application is disposed the runtimeToolingService should not be called")
  public void doNotDisposeRemoteApplication() throws Exception {
    when(runtimeToolingService.deployApplication(artifactFile)).thenReturn(APP_ID);

    DefaultApplication application =
        new DefaultApplication(applicationContentUrl, context, emptyMap());
    verifyNoMoreInteractions(runtimeToolingService);
    application.dispose();
  }

  @Test
  @Description("Validates that class loader for application should be closed once the application is disposed")
  public void disposeInvalidatesClassLoader() throws Exception {
    when(applicationDescriptor.getRootFolder()).thenReturn(artifactFile);
    when(applicationClassLoader.getArtifactDescriptor()).thenReturn(applicationDescriptor);
    when(context.getApplicationService()).thenReturn(applicationService);
    when(applicationService.createApplicationClassLoader(anyObject(), anyObject(), anyObject()))
        .thenReturn(applicationClassLoader);

    DefaultApplication application = new DefaultApplication(applicationContentUrl, context, emptyMap());

    assertThat(application.getArtifactClassLoader(), sameInstance(applicationClassLoader));
    application.dispose();

    verify(applicationService).createApplicationClassLoader(anyString(), any(File.class), any(File.class));
    verify(applicationClassLoader).dispose();
  }

  @Test
  @Description("Validates that when an application is fetched only the remote application is disposed if the URL has changed")
  public void disposeRemoteApplicationIfToolingUrlChanges() throws Exception {
    when(runtimeToolingService.deployApplication(artifactFile)).thenReturn(APP_ID);

    DefaultApplication application =
        new DefaultApplication(applicationContentUrl, context, emptyMap());
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(APP_ID));

    AgentConfiguration agentConfiguration = mock(AgentConfiguration.class);
    URL originalUrl = new URL("http://localhost:9991/mule");
    when(agentConfiguration.getToolingApiUrl()).thenReturn(originalUrl);
    Optional<AgentConfiguration> agentConfigurationOptional = Optional.of(agentConfiguration);
    when(context.getAgentConfiguration()).thenReturn(agentConfigurationOptional);

    AgentConfiguration newAgentConfiguration = mock(AgentConfiguration.class);
    URL newUrl = new URL("http://localhost:9992/mule");
    when(newAgentConfiguration.getToolingApiUrl()).thenReturn(newUrl);
    Optional<AgentConfiguration> newAgentConfigurationOptional = Optional.of(newAgentConfiguration);
    ToolingArtifactContext newContext = mock(ToolingArtifactContext.class);
    when(newContext.getAgentConfiguration()).thenReturn(newAgentConfigurationOptional);
    when(newContext.getRuntimeToolingService()).thenReturn(runtimeToolingService);

    application.setContext(newContext);

    String otherAppId = "otherAppId";
    when(runtimeToolingService.deployApplication(artifactFile)).thenReturn(otherAppId);
    applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(otherAppId));

    verify(runtimeToolingService).disposeApplication(APP_ID);
    verify(runtimeToolingService, times(2)).deployApplication(artifactFile);

  }

  @Test
  @Description("Validates that extension models are loaded for an application when applicationUrlContent points to an exploded content")
  public void getDependenciesExplodedApplicationUrl() throws Exception {
    assertExtensionModels(artifactFile, applicationContentUrl, false);
  }

  @Test
  @Description("Validates that extension models are loaded for an application when applicationUrlContent points to a URL")
  public void getDependenciesJarApplicationUrl() throws Exception {
    File jarFile = temporaryFolder.newFile();
    compress(jarFile, artifactFile);

    wireMockRule.stubFor(
                         get(urlEqualTo("/app"))
                             .willReturn(aResponse()
                                 .withStatus(200)
                                 .withHeader("Content-Type", "application/octet-stream")
                                 .withBody(IOUtils.toByteArray(jarFile.toURI().toURL()))));
    int port = wireMockRule.port();

    URL url = new URI("http://localhost:" + port + REQUEST_PATH).toURL();
    assertExtensionModels(artifactFile, url, true);
  }

  @Test
  @Description("Validates that extension models are loaded for an application when applicationUrlContent points to a Jar file, and it is not removed")
  public void getDependenciesJarApplication() throws Exception {
    URL url = artifactJarFile.toURI().toURL();
    assertExtensionModels(artifactJarFile, url, true);
    assertThat(artifactJarFile.exists(), is(true));
  }

  private void assertExtensionModels(File artifactFile, URL url, boolean useTempApplicationFile) throws IOException {
    DefaultApplication application = new DefaultApplication(url, context, emptyMap());

    List<BundleDependency> pluginDependenciesResolved = newArrayList();
    pluginDependenciesResolved.add(new BundleDependency.Builder()
        .sedBundleDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.test")
            .setArtifactId("plugin-test")
            .setClassifier("mule-plugin")
            .setVersion("1.0")
            .build())
        .build());
    pluginDependenciesResolved.add(new BundleDependency.Builder()
        .sedBundleDescriptor(new BundleDescriptor.Builder()
            .setGroupId("org.mule.test")
            .setArtifactId("plugin-dependency")
            .setVersion("1.0")
            .build())
        .build());

    List<ArtifactClassLoader> artifactPluginClassLoaders = new ArrayList<>();

    ArtifactClassLoader pluginTestClassLoader = mock(ArtifactClassLoader.class);
    ArtifactDescriptor pluginTestArtifactDescriptor = mock(ArtifactDescriptor.class);
    when(pluginTestArtifactDescriptor.getBundleDescriptor())
        .thenReturn(toBundleDescriptor(pluginDependenciesResolved.get(0).getDescriptor()));
    when(pluginTestClassLoader.getArtifactDescriptor()).thenReturn(pluginTestArtifactDescriptor);
    artifactPluginClassLoaders.add(pluginTestClassLoader);

    ToolingArtifactClassLoader toolingArtifactClassLoader = mock(ToolingArtifactClassLoader.class);
    when(toolingArtifactClassLoader.getArtifactPluginClassLoaders()).thenReturn(artifactPluginClassLoaders);
    when(applicationService.createApplicationClassLoader(anyString(), useTempApplicationFile ? any(File.class) : eq(artifactFile),
                                                         any(File.class))).thenReturn(toolingArtifactClassLoader);
    when(context.getApplicationService()).thenReturn(applicationService);
    when(context.getMuleRuntimeExtensionModelProvider()).thenReturn(runtimeExtensionModelProvider);
    when(runtimeExtensionModelProvider.getRuntimeExtensionModels()).thenReturn(newArrayList(runtimeExtensionModel));
    when(runtimeExtensionModelProvider
        .getExtensionModel(toArtifactDescriptor(pluginTestArtifactDescriptor.getBundleDescriptor())))
            .thenReturn(of(extensionModel));

    final List<ExtensionModel> extensionModels = application.getExtensionModels();

    assertThat(extensionModels, hasSize(2));
    assertThat(extensionModels.get(0), sameInstance(extensionModel));
    assertThat(extensionModels.get(1), sameInstance(runtimeExtensionModel));

    verify(runtimeExtensionModelProvider)
        .getExtensionModel(toArtifactDescriptor(pluginTestArtifactDescriptor.getBundleDescriptor()));
  }

  private org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor toBundleDescriptor(BundleDescriptor descriptor) {
    return new org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor.Builder()
        .setGroupId(descriptor.getGroupId())
        .setArtifactId(descriptor.getArtifactId())
        .setVersion(descriptor.getVersion())
        .setType(descriptor.getType())
        .setClassifier(descriptor.getClassifier().orElse(null))
        .build();
  }

  private org.mule.tooling.client.api.descriptors.ArtifactDescriptor toArtifactDescriptor(
                                                                                          org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor bundleDescriptor) {
    final org.mule.tooling.client.api.descriptors.ArtifactDescriptor.Builder builder =
        org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder()
            .withGroupId(bundleDescriptor.getGroupId())
            .withArtifactId(bundleDescriptor.getArtifactId())
            .withVersion(bundleDescriptor.getVersion())
            .withExtension(bundleDescriptor.getType());
    bundleDescriptor.getClassifier().ifPresent(classifier -> builder.withClassifier(classifier));
    return builder.build();
  }


}
