/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptyMap;
import static java.util.Optional.empty;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.core.Is.is;
import static org.hamcrest.core.IsNull.notNullValue;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.mule.maven.client.api.MavenClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.connectivity.ConnectivityTestingRequest;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.ApplicationService;
import org.mule.tooling.client.internal.application.DefaultApplication;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.util.UUID;
import java.util.concurrent.Callable;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
@Feature("ToolingRuntimeClient")
@Story("Tooling Runtime Client configuration")
public class DefaultToolingRuntimeClientTestCase {

  @Mock
  private MavenClient mavenClient;
  @Mock
  private Application application;
  @Mock
  private MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider;
  @Mock
  private ApplicationService applicationService;
  @Mock
  private ApplicationCache applicationCache;
  @Mock
  private DslSyntaxServiceCache dslSyntaxServiceCache;
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public ExpectedException expectedException = none();

  private File appLocation = toFile(this.getClass().getClassLoader().getResource("applications/email"));

  @Test
  @Description("Validates that a new ToolingArtifact is created when not present in cache")
  public void newToolingArtifact() throws IOException {
    when(applicationCache.getApplication(any(String.class), any(Callable.class))).thenReturn(application);

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, empty(), muleRuntimeExtensionModelProvider,
                                        applicationService, applicationCache,
                                        empty(), dslSyntaxServiceCache);
    ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(temporaryFolder.newFolder().toURI().toURL(), emptyMap());
    assertThat(toolingArtifact, is(notNullValue()));

    verify(applicationCache).getApplication(any(String.class), any(Callable.class));
  }

  @Test
  @Description("Validates that a ToolingArtifact is invalidated from cache whenever is disposed")
  public void onDisposeToolingArtifactIsInvalidatedFromCache() throws IOException {
    when(applicationCache.getApplication(any(String.class), any(Callable.class))).thenReturn(application);

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, empty(), muleRuntimeExtensionModelProvider,
                                        applicationService, applicationCache,
                                        empty(), dslSyntaxServiceCache);
    ToolingArtifact toolingArtifact =
        toolingRuntimeClient.newToolingArtifact(temporaryFolder.newFolder().toURI().toURL(), emptyMap());
    assertThat(toolingArtifact, is(notNullValue()));

    doNothing().when(applicationCache).invalidate(toolingArtifact.getId());
    toolingArtifact.dispose();

    verify(applicationCache).getApplication(any(String.class), any(Callable.class));
    verify(applicationCache).invalidate(toolingArtifact.getId());
  }

  @Test
  @Description("Validates that a ToolingArtifact can be fetched and the application context is set")
  public void fetchToolingArtifact() throws IOException {
    String id = UUID.randomUUID().toString();
    when(applicationCache.getApplication(eq(id), any(Callable.class))).thenReturn(application);

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, empty(), muleRuntimeExtensionModelProvider,
                                        applicationService, applicationCache,
                                        empty(), dslSyntaxServiceCache);
    ToolingArtifact toolingArtifact = toolingRuntimeClient.fetchToolingArtifact(id);
    assertThat(toolingArtifact, is(notNullValue()));

    verify(applicationCache).getApplication(eq(id), any(Callable.class));
    verify(application).setContext(any(ToolingArtifactContext.class));
  }

  @Test
  @Description("Validates that a ToolingArtifact is fetched and no longer in cache an exception is thrown")
  public void fetchToolingArtifactMissedCache() throws IOException {
    String id = UUID.randomUUID().toString();
    when(applicationCache.getApplication(eq(id), any(Callable.class)))
        .then(invocationOnMock -> ((Callable) invocationOnMock.getArguments()[1]).call());

    DefaultToolingRuntimeClient toolingRuntimeClient =
        new DefaultToolingRuntimeClient(mavenClient, empty(), muleRuntimeExtensionModelProvider,
                                        applicationService, applicationCache,
                                        empty(), dslSyntaxServiceCache);

    expectedException.expect(ToolingArtifactNotFoundException.class);
    toolingRuntimeClient.fetchToolingArtifact(id);
  }

  @Test
  @Description("Checks that when tooling artifact has to be deployed due to a connectivity testing request if no configuration for the remote tooling service is set an exception is thrown")
  public void shouldFailWhenTryingToConnectWithoutConfiguration() throws Exception {
    DefaultToolingArtifactContext context = new DefaultToolingArtifactContext();
    context.setAgentConfiguration(empty());

    URL applicationUrlContent = toUrl(appLocation.toURI());
    DefaultToolingRuntimeClient client =
        new DefaultToolingRuntimeClient(mavenClient, empty(),
                                        muleRuntimeExtensionModelProvider,
                                        applicationService,
                                        applicationCache, empty(), dslSyntaxServiceCache);
    when(applicationCache.getApplication(any(String.class), any(Callable.class)))
        .thenReturn(new DefaultApplication(applicationUrlContent, context, emptyMap()));

    expectedException.expect(MissingToolingConfigurationException.class);
    client.newToolingArtifact(applicationUrlContent, emptyMap()).connectivityTestingService()
        .testConnection(new ConnectivityTestingRequest());
  }

  private URL toUrl(URI uri) {
    try {
      return uri.toURL();
    } catch (MalformedURLException e) {
      throw new RuntimeException("Error while getting URL", e);
    }
  }

}
