/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.base.Throwables.propagateIfPossible;
import static java.lang.String.format;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static org.apache.commons.io.FileUtils.toFile;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.DataSensePartsFactory.toDataSenseComponentInfoDTO;
import static org.mule.tooling.client.internal.DataSensePartsFactory.toDataSenseInfoDTO;
import static org.mule.tooling.client.internal.DataSenseResolveFactory.fromDataSenseResolutionScopeDTO;
import static org.mule.tooling.client.internal.LocationFactory.fromLocationDTO;
import static org.mule.tooling.client.internal.serialization.XStreamServerSerializer.deserialize;
import static org.mule.tooling.client.internal.serialization.XStreamServerSerializer.serialize;
import org.mule.datasense.api.DataSense;
import org.mule.datasense.api.DataSenseInfo;
import org.mule.datasense.api.metadataprovider.ApplicationModel;
import org.mule.datasense.api.metadataprovider.CachedDataSenseProvider;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataCacheProvider;
import org.mule.datasense.api.metadataprovider.DataSenseProvider;
import org.mule.datasense.api.metadataprovider.DefaultApplicationModel;
import org.mule.datasense.api.metadataprovider.DefaultDataSenseProvider;
import org.mule.datasense.api.metadataprovider.ExtensionsProvider;
import org.mule.datasense.impl.DefaultDataSense;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.tooling.client.api.datasense.DataSenseComponentInfo;
import org.mule.tooling.client.api.datasense.DataSenseRequest;
import org.mule.tooling.client.api.datasense.DataSenseService;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.exception.MissingToolingConfigurationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.application.Application;

import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableSet;

import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Default implementation of {@link DataSenseService}.
 *
 * @since 4.0
 */
public class DefaultDataSenseService implements DataSenseService, Command {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private static final String APP_TYPES_DATA = "application-types.xml";
  private static final String APP_TYPES_TEST_DATA = "application-types-test.xml";

  private DataSense dataSense = new DefaultDataSense();

  private Application application;
  private MetadataProvider metadataProvider;
  private MetadataCache metadataCache;

  private AtomicBoolean disposed = new AtomicBoolean(false);

  /**
   * Creates an instance of the DataSense service.
   *
   * @param application {@link Application} to resolve DataSense. Non null.
   * @param metadataProvider {@link MetadataProvider} to resolve dynamic Metadata if needed. Non null.
   * @param metadataCache {@link MetadataCache}. Non null.
   */
  public DefaultDataSenseService(Application application, MetadataProvider metadataProvider, MetadataCache metadataCache) {
    checkNotNull(application, "application cannot be null");
    checkNotNull(metadataProvider, "metadataProvider cannot be null");
    checkNotNull(metadataCache, "metadataCache cannot be null");

    this.application = application;
    this.metadataProvider = metadataProvider;
    this.metadataCache = metadataCache;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<org.mule.tooling.client.api.datasense.DataSenseInfo> resolveDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    return toDataSenseInfoDTO(internalResolveDataSense(dataSenseRequest));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<DataSenseComponentInfo> resolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    return toDataSenseComponentInfoDTO(internalResolveComponentDataSense(dataSenseRequest));
  }

  private Optional<DataSenseInfo> internalResolveDataSense(DataSenseRequest dataSenseRequest) {
    checkState();

    Location location = fromLocationDTO(dataSenseRequest.getLocation());
    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider =
        new DataSenseMetadataCacheAdapter(metadataCache, application.getProperties());

    if (logger.isDebugEnabled()) {
      logger.debug(format("Resolving DataSense for component location: %s on %s", dataSenseRequest.getLocation(), application));
    }

    try {
      ApplicationModel applicationModel = buildDataSenseApplicationModel(application);

      DataSenseProvider dataSenseProvider =
          getDataSenseProvider(applicationModel,
                               () -> ImmutableSet.<ExtensionModel>builder().addAll(application.getExtensionModels()).build(),
                               of(dataSenseMetadataCacheProvider),
                               dataSenseRequest);


      return withContextClassLoader(application.getArtifactClassLoader().getClassLoader(),
                                    () -> this.dataSense.resolve(fromDataSenseResolutionScopeDTO(dataSenseRequest
                                        .getDataSenseResolutionScope()), applicationModel,
                                                                 dataSenseProvider));
    } catch (Exception e) {
      propagateIfPossible(e, MissingToolingConfigurationException.class);
      throw new ToolingException(format("Error while resolving DataSense for location: %s on %s", location, application), e);
    }
  }

  private Optional<org.mule.datasense.api.DataSenseComponentInfo> internalResolveComponentDataSense(DataSenseRequest dataSenseRequest) {
    checkState();
    Location location = fromLocationDTO(dataSenseRequest.getLocation());
    checkNotNull(location, "location cannot be null");

    DataSenseMetadataCacheProvider dataSenseMetadataCacheProvider =
        new DataSenseMetadataCacheAdapter(metadataCache, application.getProperties());

    if (logger.isDebugEnabled()) {
      logger.debug(format("Resolving DataSense for location: %s on %s", dataSenseRequest.getLocation(), application));
    }

    try {
      ApplicationModel applicationModel = buildDataSenseApplicationModel(application);
      DataSenseProvider dataSenseProvider =
          getDataSenseProvider(applicationModel,
                               () -> ImmutableSet.<ExtensionModel>builder().addAll(application.getExtensionModels()).build(),
                               of(dataSenseMetadataCacheProvider),
                               dataSenseRequest);

      return withContextClassLoader(application.getArtifactClassLoader().getClassLoader(),
                                    () -> this.dataSense.resolveComponent(location, applicationModel, dataSenseProvider));
    } catch (Exception e) {
      propagateIfPossible(e, MissingToolingConfigurationException.class);
      throw new ToolingException(format("Error while resolving DataSense for location %s on %s", location, application), e);
    }
  }

  public void dispose() {
    disposed.compareAndSet(false, true);
  }

  private ApplicationModel buildDataSenseApplicationModel(Application application) {
    String applicationName = application.getApplicationName();
    org.mule.runtime.config.internal.model.ApplicationModel applicationModel = application.getApplicationModel();

    List<String> typesDataList = new ArrayList<>();
    getApplicationTypes(APP_TYPES_DATA, application).ifPresent(typesData -> typesDataList.add(typesData));
    getApplicationTypes(APP_TYPES_TEST_DATA, application).ifPresent(typesData -> typesDataList.add(typesData));
    return new DefaultApplicationModel(applicationName,
                                       applicationModel,
                                       typesDataList,
                                       resource -> ofNullable(
                                                              application.getArtifactClassLoader().getClassLoader()
                                                                  .getResource(resource)).map(url -> toFile(url).toURI()));
  }

  private Optional<String> getApplicationTypes(String resource, Application application) {
    InputStream appTypeData = application.getArtifactClassLoader().getClassLoader().getResourceAsStream(resource);
    String typesData = null;
    if (appTypeData != null) {
      logger.debug("Found application custom types data from: " + resource);
      try {
        typesData = IOUtils.toString(appTypeData);
      } catch (IOException e) {
        throw new ToolingException(format("Error while reading application custom types file: %s for %s", resource, application),
                                   e);
      }
    }
    return ofNullable(typesData);
  }

  private DataSenseProvider getDataSenseProvider(ApplicationModel applicationModel, ExtensionsProvider extensionsProvider,
                                                 Optional<DataSenseMetadataCacheProvider> dataSenseMetadataCacheProvider,
                                                 DataSenseRequest request) {
    DataSenseProvider dataSenseProvider =
        new DefaultDataSenseProvider(extensionsProvider, new ToolingDataSenseMetadataProvider(metadataProvider,
                                                                                              request.getMetadataTimeout()));
    if (dataSenseMetadataCacheProvider.isPresent()) {
      dataSenseProvider = new CachedDataSenseProvider(applicationModel, dataSenseMetadataCacheProvider.get(), dataSenseProvider);
    }

    return dataSenseProvider;
  }

  private void checkState() {
    Preconditions.checkState(!disposed.get(), "Service already disposed, cannot be used anymore");
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "resolveDataSense": {
        org.mule.runtime.api.util.Preconditions.checkState(arguments.length == 1,
                                                           format("Wrong number of arguments when invoking method created on %s",
                                                                  this.getClass().getName()));
        org.mule.runtime.api.util.Preconditions
            .checkState(classes.length == 1 && classes[0].equals(DataSenseRequest.class.getName()),
                        format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serialize(resolveDataSense(deserialize(arguments[0])));
      }
      case "resolveComponentDataSense": {
        org.mule.runtime.api.util.Preconditions.checkState(arguments.length == 1,
                                                           format("Wrong number of arguments when invoking method created on %s",
                                                                  this.getClass().getName()));
        org.mule.runtime.api.util.Preconditions
            .checkState(classes.length == 1 && classes[0].equals(DataSenseRequest.class.getName()),
                        format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return serialize(resolveComponentDataSense(deserialize(arguments[0])));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
