/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static java.lang.String.format;
import static java.util.UUID.randomUUID;
import static org.mule.runtime.api.util.Preconditions.checkState;
import static org.mule.tooling.client.internal.Command.methodNotFound;
import static org.mule.tooling.client.internal.serialization.XStreamServerSerializer.deserialize;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.util.LazyValue;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.api.ToolingRuntimeClient;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.artifact.declaration.ArtifactSerializationService;
import org.mule.tooling.client.api.artifact.dsl.DslSyntaxResolverService;
import org.mule.tooling.client.api.configuration.agent.AgentConfiguration;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.datasense.MetadataCacheFactory;
import org.mule.tooling.client.api.exception.ToolingArtifactNotFoundException;
import org.mule.tooling.client.api.extension.ExtensionModelService;
import org.mule.tooling.client.api.icons.ExtensionIconsService;
import org.mule.tooling.client.api.message.history.MessageHistoryService;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.client.internal.application.ApplicationService;
import org.mule.tooling.client.internal.application.DefaultApplication;
import org.mule.tooling.client.internal.artifact.DefaultArtifactSerializationService;
import org.mule.tooling.client.internal.dsl.DefaultDslSyntaxResolverService;
import org.mule.tooling.client.internal.dsl.DslSyntaxServiceCache;
import org.mule.tooling.client.internal.icons.DefaultExtensionIconsService;

import java.net.URL;
import java.util.Map;
import java.util.Optional;

/**
 * Implementation of a {@link ToolingRuntimeClient}. It uses SPI to look up for {@link RuntimeToolingService} implementation to
 * resolve tooling operations.
 *
 * @since 4.0
 */
public class DefaultToolingRuntimeClient implements ToolingRuntimeClient, Command {

  private DefaultToolingArtifactContext context;

  private ExtensionModelService extensionModelService;
  private ArtifactSerializationService artifactSerializationService;
  private DslSyntaxResolverService dslSyntaxResolverService;
  private MessageHistoryService messageHistoryService;
  private ExtensionIconsService extensionIconsService;

  /**
   * Creates an instance of the client.
   *
   * @param mavenClient {@link MavenClient} to resolve dependencies. Non null.
   * @param agentConfigurationOptional {@link Optional} Mule Agent configuration. Non null.
   * @param muleRuntimeExtensionModelProvider provider to load {@link org.mule.runtime.api.meta.model.ExtensionModel}s. Non null.
   * @param applicationService {@link ApplicationService} to create the
   *        {@link org.mule.tooling.client.internal.application.Application} class loader. Non null.
   * @param applicationCache {@link ApplicationCache} to fetch or create
   *        {@link org.mule.tooling.client.internal.application.DefaultApplication}. Non null.
   * @param metadataCacheFactoryOptional {@link Optional} {@link MetadataCacheFactory} to be used when resolving Metadata during data sense resolution. Non null.
   * @param dslSyntaxServiceCache {@link DslSyntaxServiceCache}. Non null.
   */
  public DefaultToolingRuntimeClient(MavenClient mavenClient,
                                     Optional<AgentConfiguration> agentConfigurationOptional,
                                     MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider,
                                     ApplicationService applicationService, ApplicationCache applicationCache,
                                     Optional<MetadataCacheFactory> metadataCacheFactoryOptional,
                                     DslSyntaxServiceCache dslSyntaxServiceCache) {
    checkNotNull(mavenClient, "aetherDependencyResolver cannot be null");
    checkNotNull(agentConfigurationOptional, "agentConfigurationOptional cannot be null");
    checkNotNull(muleRuntimeExtensionModelProvider, "muleRuntimeExtensionModelProvider cannot be null");
    checkNotNull(applicationService, "applicationService cannot be null");
    checkNotNull(applicationCache, "applicationCache cannot be null");
    checkNotNull(metadataCacheFactoryOptional, "metadataCacheFactoryOptional cannot be null");
    checkNotNull(dslSyntaxServiceCache, "dslSyntaxServiceCache cannot be null");

    context = new DefaultToolingArtifactContext();
    context.setAgentConfiguration(agentConfigurationOptional);
    context.setMavenClient(mavenClient);
    context.setMuleRuntimeExtensionModelProvider(muleRuntimeExtensionModelProvider);
    context.setApplicationService(applicationService);
    context.setApplicationCache(applicationCache);
    context.setMetadataCacheFactory(metadataCacheFactoryOptional);

    extensionModelService = new ToolingExtensionModelAdapter(muleRuntimeExtensionModelProvider);
    artifactSerializationService =
        new DefaultArtifactSerializationService(muleRuntimeExtensionModelProvider.getRuntimeExtensionModels(),
                                                muleRuntimeExtensionModelProvider,
                                                mavenClient);
    messageHistoryService = new DefaultMessageHistoryService(new LazyValue<>(() -> context.getRuntimeToolingService()));
    extensionModelService = new ToolingExtensionModelAdapter(muleRuntimeExtensionModelProvider);
    extensionIconsService = new DefaultExtensionIconsService(context.getMavenClient());
    dslSyntaxResolverService =
        new DefaultDslSyntaxResolverService(dslSyntaxServiceCache, mavenClient, muleRuntimeExtensionModelProvider);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ExtensionModelService extensionModelService() {
    return extensionModelService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ArtifactSerializationService artifactSerializationService() {
    return artifactSerializationService;
  }

  @Override
  public ExtensionIconsService iconsService() {
    return extensionIconsService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DslSyntaxResolverService dslSyntaxResolverService() {
    return dslSyntaxResolverService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MessageHistoryService messageHistoryService() {
    return messageHistoryService;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingArtifact newToolingArtifact(URL applicationUrlContent, Map<String, String> properties) {
    String id = randomUUID().toString();
    ApplicationCache applicationCache = context.getApplicationCache();
    Application application =
        applicationCache.getApplication(id, () -> new DefaultApplication(applicationUrlContent, context, properties));
    MetadataCache metadataCache = context.getMetadataCacheFactory()
        .map(factory -> factory.createMetadataCache(id, applicationUrlContent, application.getProperties()))
        .orElse(NoOpMetadataCache.INSTANCE);
    return newToolingArtifact(id, applicationCache, metadataCache, application);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ToolingArtifact fetchToolingArtifact(String id) throws ToolingArtifactNotFoundException {
    Application application = context.getApplicationCache().getApplication(id, () -> {
      throw new ToolingArtifactNotFoundException(
                                                 format("ToolingArtifact not found in cache for id: %s", id));
    });
    MetadataCache metadataCache =
        context.getMetadataCacheFactory()
            .map(factory -> factory.createMetadataCache(id, application.getApplicationUrlContent(), application.getProperties()))
            .orElse(NoOpMetadataCache.INSTANCE);
    application.setContext(context);
    return newToolingArtifact(id, context.getApplicationCache(), metadataCache, application);
  }

  private ToolingArtifact newToolingArtifact(final String id, final ApplicationCache applicationCache,
                                             final MetadataCache metadataCache, final Application application) {
    return new ToolingArtifactWrapper(new DefaultToolingArtifact(id, metadataCache, application) {

      @Override
      public void dispose() {
        applicationCache.invalidate(id);
      }
    });
  }

  @Override
  public Object invokeMethod(String methodName, String[] classes, String[] arguments) {
    switch (methodName) {
      case "messageHistoryService": {
        return this.messageHistoryService();
      }
      case "extensionModelService": {
        return this.extensionModelService();
      }
      case "artifactSerializationService": {
        return this.artifactSerializationService();
      }
      case "dslSyntaxResolverService": {
        return this.dslSyntaxResolverService();
      }
      case "iconsService": {
        return this.iconsService();
      }
      case "newToolingArtifact": {
        checkState(arguments.length == 2,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 2 && classes[0].equals(URL.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 2 && classes[1].equals(Map.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return newToolingArtifact(deserialize(arguments[0]), deserialize(arguments[1]));
      }
      case "fetchToolingArtifact": {
        checkState(arguments.length == 1,
                   format("Wrong number of arguments when invoking method created on %s", this.getClass().getName()));
        checkState(classes.length == 1 && classes[0].equals(String.class.getName()),
                   format("Wrong type of arguments when invoking method created on %s", this.getClass().getName()));
        return fetchToolingArtifact(deserialize(arguments[0]));
      }
    }
    throw methodNotFound(this.getClass(), methodName);
  }

}
