/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.mule.runtime.api.metadata.resolving.MetadataFailure.Builder.newFailure;
import static org.mule.runtime.api.metadata.resolving.MetadataResult.failure;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.internal.application.RemoteApplicationInvoker;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Internal implementation of {@link MetadataProvider}, that is used with DataSense and uses Mule-API objects directly.
 *
 * @since 4.0
 */
public class InternalMetadataProvider implements MetadataProvider {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private RemoteApplicationInvoker remoteApplicationInvoker;

  /**
   * Creates an instance of the provider.
   *
   * @param remoteApplicationInvoker {@link RemoteApplicationInvoker} to deploy and resolve operations against a Mule Runtime. Non null.
   */
  public InternalMetadataProvider(RemoteApplicationInvoker remoteApplicationInvoker) {
    checkNotNull(remoteApplicationInvoker, "remoteApplicationInvoker cannot be null");

    this.remoteApplicationInvoker = remoteApplicationInvoker;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest)
      throws ServiceUnavailableException, ToolingException {
    if (logger.isDebugEnabled()) {
      logger.debug("Getting metadata keys for: {} on application: {}", metadataKeysRequest, remoteApplicationInvoker);
    }

    try {
      return remoteApplicationInvoker.evaluateWithRemoteApplication((applicationId,
                                                                     runtimeToolingService) -> runtimeToolingService
                                                                         .getMetadataKeys(applicationId,
                                                                                          metadataKeysRequest.getLocation()
                                                                                              .toString(),
                                                                                          metadataKeysRequest
                                                                                              .getRequestTimeout()));
    } catch (Exception e) {
      return failure(newFailure(e).onKeys());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(MetadataRequest metadataRequest) {
    if (logger.isDebugEnabled()) {
      logger.debug("Getting operation metadata for: {} on application: {}", metadataRequest, remoteApplicationInvoker);
    }

    try {
      return remoteApplicationInvoker.evaluateWithRemoteApplication((applicationId,
                                                                     runtimeToolingService) -> runtimeToolingService
                                                                         .getOperationMetadata(applicationId,
                                                                                               metadataRequest.getLocation()
                                                                                                   .toString(),
                                                                                               metadataRequest
                                                                                                   .getRequestTimeout()));
    } catch (Exception e) {
      return failure(newFailure(e).onComponent());
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(MetadataRequest metadataRequest) {
    if (logger.isDebugEnabled()) {
      logger.debug("Getting source metadata for: {} on application: {}", metadataRequest, remoteApplicationInvoker);
    }

    try {
      return remoteApplicationInvoker.evaluateWithRemoteApplication((applicationId,
                                                                     runtimeToolingService) -> runtimeToolingService
                                                                         .getSourceMetadata(applicationId,
                                                                                            metadataRequest.getLocation()
                                                                                                .toString(),
                                                                                            metadataRequest.getRequestTimeout()));
    } catch (Exception e) {
      return failure(newFailure(e).onComponent());
    }
  }

}
