/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;


import static org.mule.tooling.client.internal.LocationFactory.toLocationDTO;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataProvider;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataFailure;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ToolingDataSenseMetadataProvider implements DataSenseMetadataProvider {

  private final Logger logger = LoggerFactory.getLogger(this.getClass());

  private MetadataResult failure = null;

  private MetadataProvider metadataProvider;
  private long metadataTimeout;

  public ToolingDataSenseMetadataProvider(MetadataProvider metadataProvider,
                                          long metadataTimeout) {
    this.metadataProvider = metadataProvider;
    this.metadataTimeout = metadataTimeout;
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(Location location) {
    final MetadataKeysRequest metadataKeysRequest = new MetadataKeysRequest();
    if (metadataTimeout > 0) {
      metadataKeysRequest.setRequestTimeout(metadataTimeout);
    }
    metadataKeysRequest.setLocation(toLocationDTO(location));
    if (failure != null) {
      return failure;
    }
    try {
      return metadataProvider.getMetadataKeys(metadataKeysRequest);
    } catch (Exception e) {
      logger.debug("Error while resolving Metadata Keys for location: " + location, e);
      failure = MetadataResult.failure(MetadataFailure.Builder.newFailure(e).onKeys());
      return failure;
    }
  }

  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(Location location) {
    if (failure != null) {
      return failure;
    }
    try {
      return metadataProvider.getOperationMetadata(getMetadataRequest(location));
    } catch (Exception e) {
      logger.debug("Error while resolving Operation Metadata for location: " + location, e);
      failure = MetadataResult.failure(MetadataFailure.Builder.newFailure(e).onComponent());
      return failure;
    }
  }

  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(Location location) {
    if (failure != null) {
      return failure;
    }
    try {
      return metadataProvider.getSourceMetadata(getMetadataRequest(location));
    } catch (Exception e) {
      logger.debug("Error while resolving Source Metadata for location: " + location, e);
      failure = MetadataResult.failure(MetadataFailure.Builder.newFailure(e).onComponent());
      return failure;
    }
  }

  private MetadataProvider.MetadataRequest getMetadataRequest(Location location) {
    MetadataProvider.MetadataRequest request = new MetadataProvider.MetadataRequest();
    request.setLocation(toLocationDTO(location));
    if (metadataTimeout > 0) {
      request.setRequestTimeout(metadataTimeout);
    }
    return request;
  }

}
