/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dataweave;

import static java.util.Objects.requireNonNull;
import static org.mule.runtime.api.exception.ExceptionHelper.getRootException;
import static org.mule.runtime.api.metadata.DataType.fromFunction;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;
import static org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse.builder;
import org.mule.runtime.api.el.BindingContext;
import org.mule.runtime.api.el.ExpressionFunction;
import org.mule.runtime.api.metadata.DataType;
import org.mule.runtime.api.metadata.MediaType;
import org.mule.runtime.api.metadata.TypedValue;
import org.mule.runtime.api.streaming.bytes.CursorStream;
import org.mule.runtime.api.streaming.bytes.CursorStreamProvider;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.core.internal.el.mvel.function.PropertyAccessFunction;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewRequest;
import org.mule.tooling.client.api.dataweave.DataWeavePreviewResponse;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.application.Application;
import org.mule.tooling.event.model.EventModel;
import org.mule.tooling.event.model.MessageModel;
import org.mule.tooling.event.model.TypedValueModel;
import org.mule.weave.v2.el.WeaveExpressionLanguage;

import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.Map;

import org.apache.commons.io.IOUtils;

/**
 * {@link DataWeaveRunner} that executes an script locally using DataWeave API.
 *
 *
 * @since 4.0
 */
public class LocalRunner implements DataWeaveRunner {

  private static final String APPLICATION_JAVA = "application/java";
  private static final String DEFAULT_ATTRIBUTES_MEDIA_TYPE = "application/java";

  private static final String VARIABLES_ID = "vars";
  private static final String ATTRIBUTES_ID = "attributes";
  private static final String PAYLOAD_ID = "payload";

  private Application application;

  /**
   * Creates an instance of the runner with the given {@link Application}.
   *
   * @param application the {@link Application} to get its class loader.
   */
  public LocalRunner(Application application) {
    requireNonNull(application, "application cannot be null");
    this.application = application;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public DataWeavePreviewResponse execute(DataWeavePreviewRequest parameters) {
    return withContextClassLoader(application.getArtifactClassLoader().getClassLoader(), () -> {
      try {
        WeaveExpressionLanguage executor = new WeaveExpressionLanguage();
        TypedValue response = executor.evaluate(getScript(parameters), buildContextFromParameters(parameters));

        return builder().result(getResult(response)).mediaType(getMediaType(response)).valid(true).build();
      } catch (Exception e) {
        return builder().errorMessage(getRootException(e).getMessage()).build();
      }
    });
  }

  private String getMediaType(TypedValue evaluate) {
    DataType dataType = evaluate.getDataType();
    MediaType mediaType = dataType.getMediaType();
    return mediaType.withoutParameters().toString();
  }

  private String getResult(TypedValue result) {
    final Object value = result.getValue();
    if (value instanceof CursorStreamProvider) {
      return manageCursorStream(result);
    } else {
      return serializeObject(value);
    }
  }

  protected String manageCursorStream(TypedValue<CursorStreamProvider> result) {
    CursorStreamProvider streamProvider = null;
    try {
      streamProvider = result.getValue();
      Charset encoding = result.getDataType().getMediaType().getCharset().orElseGet(Charset::defaultCharset);
      try (CursorStream cursorStream = streamProvider.openCursor()) {
        return IOUtils.toString(cursorStream, encoding);
      }
    } catch (IOException e) {
      throw new ToolingException("DataWeave :: Error consuming DataWeave result", e);
    } finally {
      if (streamProvider != null) {
        streamProvider.close();
      }
    }
  }

  protected String serializeObject(Object value) {
    Gson gson = new GsonBuilder().setPrettyPrinting().create();
    return gson.toJson(value);
  }

  private BindingContext buildContextFromParameters(DataWeavePreviewRequest parameters) throws IOException {
    BindingContext.Builder bindingBuilder = BindingContext.builder();

    final EventModel event = getEvent(parameters);
    if (event != null) {
      manageVariables(bindingBuilder, event);
      manageMessage(bindingBuilder, event);
    }

    appendFunctions(application.getApplicationModel(), bindingBuilder);

    return bindingBuilder.build();
  }

  private void appendFunctions(ApplicationModel applicationModel, BindingContext.Builder builder) {
    PropertyAccessFunction propertyFunction = new PropertyAccessFunction(applicationModel.getConfigurationProperties());
    builder.addBinding("p", new TypedValue(propertyFunction, fromFunction(propertyFunction)));

    ExpressionFunction lookupFunction = new MockLookupFunction();
    builder.addBinding("lookup", new TypedValue(lookupFunction, fromFunction(lookupFunction)));

    ExpressionFunction causedByFunction = new MockCausedByFunction();
    builder.addBinding("causedBy", new TypedValue(causedByFunction, fromFunction(causedByFunction)));

  }

  private void manageMessage(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    final MessageModel message = event.getMessage();
    if (message != null) {
      if (message.getAttributes() != null) {
        bindingBuilder.addBinding(ATTRIBUTES_ID, asTypedValue(message.getAttributes()));
      }

      if (message.getPayload() != null) {
        bindingBuilder.addBinding(PAYLOAD_ID, asTypedValue(message.getPayload()));
      }
    }
  }

  private void manageVariables(BindingContext.Builder bindingBuilder, EventModel event) throws IOException {
    if (event.getVariables() != null && !event.getVariables().isEmpty()) {
      Map<String, TypedValue> variables = new HashMap<>();
      for (Map.Entry<String, TypedValueModel> pair : event.getVariables().entrySet()) {
        variables.put(pair.getKey(), asTypedValue(pair.getValue()));
      }
      DataType variablesDataType = DataType.builder().type(Map.class).mediaType(DEFAULT_ATTRIBUTES_MEDIA_TYPE).build();
      bindingBuilder.addBinding(VARIABLES_ID, new TypedValue<>(variables, variablesDataType));
    }
  }

  protected TypedValue<?> asTypedValue(TypedValueModel restTypedValue) throws IOException {
    String mediaType = restTypedValue.getDataType().getMediaType();
    if (APPLICATION_JAVA.equals(mediaType)) {
      throw new IllegalArgumentException("Java input not supported, serialize to DW script");
    } else {
      DataType dataType = DataType.builder().type(String.class).mediaType(mediaType).build();
      return new TypedValue<>(new String(restTypedValue.getContent(), "UTF-8"), dataType);
    }
  }

  private EventModel getEvent(DataWeavePreviewRequest parameters) {
    return parameters.getEvent();
  }

  private String getScript(DataWeavePreviewRequest parameters) {
    return parameters.getScript();
  }
}
