/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.google.common.collect.Lists.newArrayList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singleton;
import static java.util.Collections.singletonList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static jersey.repackaged.com.google.common.collect.Sets.newHashSet;
import static org.hamcrest.CoreMatchers.hasItem;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertThat;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mule.runtime.api.meta.model.display.PathModel.Location.EMBEDDED;
import static org.mule.runtime.api.meta.model.display.PathModel.Type.FILE;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.CONFIG;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.FLOW;
import static org.mule.runtime.extension.api.stereotype.MuleStereotypes.OBJECT_STORE;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.java.api.JavaTypeLoader;
import org.mule.runtime.api.meta.Category;
import org.mule.runtime.api.meta.ExpressionSupport;
import org.mule.runtime.api.meta.ExternalLibraryType;
import org.mule.runtime.api.meta.model.ExternalLibraryModel;
import org.mule.runtime.api.meta.model.ParameterDslConfiguration;
import org.mule.runtime.api.meta.model.XmlDslModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionManagementType;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.display.DisplayModel;
import org.mule.runtime.api.meta.model.display.PathModel;
import org.mule.runtime.api.meta.model.error.ImmutableErrorModel;
import org.mule.runtime.api.meta.model.parameter.ParameterRole;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModel;
import org.mule.runtime.api.meta.model.stereotype.StereotypeModelBuilder;
import org.mule.runtime.extension.api.model.construct.ImmutableConstructModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedChainModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedComponentModel;
import org.mule.runtime.extension.api.model.nested.ImmutableNestedRouteModel;
import org.mule.tooling.client.api.extension.model.ErrorModel;
import org.mule.tooling.client.api.extension.model.ExtensionModel;
import org.mule.tooling.client.api.extension.model.construct.ConstructModel;
import org.mule.tooling.client.api.extension.model.function.FunctionModel;
import org.mule.tooling.client.api.extension.model.nested.NestedChainModel;
import org.mule.tooling.client.api.extension.model.nested.NestedComponentModel;
import org.mule.tooling.client.api.extension.model.nested.NestedRouteModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterGroupModel;
import org.mule.tooling.client.api.extension.model.parameter.ParameterModel;

import com.google.common.collect.ImmutableList;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.hamcrest.collection.IsCollectionWithSize;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import scala.Int;

@Feature("DTOs")
@Story("Extension Model for Tooling API")
@RunWith(MockitoJUnitRunner.class)
public class ExtensionModelFactoryTestCase {

  private static final String EXTENSION_NAME = "name";
  private static final String EXTENSION_DESCRIPTION = "description";
  private static final String EXTENSION_VERSION = "1.0";
  private static final String EXTENSION_VENDOR = "MuleSoft";

  private static final String DISPLAY_NAME = "displayName";
  private static final String DISPLAY_EXAMPLE = "displayExample";
  private static final String DISPLAY_SUMMARY = "displaySummary";
  private static final List<String> FILE_EXTENSIONS = newArrayList("java");

  private static final String PREFIX = "prefix";
  private static final String NAMESPACE = "namespace";
  private static final String SCHEMA_VERSION = "schemaVersion";
  private static final String SCHEMA_LOCATION = "schemaLocation";

  private static final String CONFIGURATION_NAME = "configName";
  private static final String CONFIGURATION_DESCRIPTION = "configDescription";

  private static final String CONNECTION_PROVIDER_NAME = "connectionProviderName";
  private static final String CONNECTION_PROVIDER_DESCRIPTION = "connectionProviderDescription";

  private static final String EXTERNAL_LIB_MODEL_NAME = "externalLibModelName";
  private static final String EXTERNAL_LIB_MODEL_DESCRIPTION = "externalLibModelDescription";
  private static final String EXTERNAL_LIB_MODEL_REGEXP_MATCHER = "a-z";

  private static final String FUNCTION_NAME = "functionName";

  private static final String SIMPLE_PARAMETER_NAME = "parameter";
  private static final String PARAMETER_WITH_REFERENCES_NAME = "withRefs";
  private static final String SIMPLE_PARAMETER_GROUP_NAME = "parameterGroup";

  private static final String ERROR_MODEL_TYPE = "type";
  private static final String PARENT_ERROR_MODEL_TYPE = "parent-type";
  private static final String ERROR_MODEL_NAMESPACE = "namespace";
  private static final String PARENT_ERROR_MODEL_NAMESPACE = "parent-namespace";

  private static final String CONSTRUCT_NAME = "constructName";
  private static final String CONSTRUCT_DESCRIPTION = "constructDescription";

  private org.mule.runtime.api.meta.model.ExtensionModel runtimeExtensionModel;
  private XmlDslModel xmlDslModel;
  private DisplayModel displayModel;
  private ExternalLibraryModel externalLibraryModel;
  private MetadataType stringType;

  private static final String MIN_MULE_VERSION = "4.0.0";

  @Mock
  org.mule.runtime.api.meta.model.function.FunctionModel runtimeFunctionModel;

  @Mock
  org.mule.runtime.api.meta.model.parameter.ParameterModel simpleRuntimeParameterModel;

  @Mock
  org.mule.runtime.api.meta.model.parameter.ParameterGroupModel simpleRuntimeParameterGroupModel;

  @Before
  public void mockExtensionModelBase() {
    stringType = new JavaTypeLoader(Thread.currentThread().getContextClassLoader()).load(String.class);

    initialiseExtensionModel();
    initialiseSimpleParameterGroup();
    initialiseFunction(runtimeFunctionModel);

    when(runtimeExtensionModel.getFunctionModels()).thenReturn(singletonList(runtimeFunctionModel));
  }

  @Test
  public void testExtensionModel() {
    ExtensionModel dto = new ExtensionModelFactory().createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertExtensionModelBasicAttributes(dto);
  }

  @Test
  public void testParameterWithConfigReferences() {
    assertParameterAllowedStereotype(CONFIG);
  }

  @Test
  public void testParameterWithFlowReferences() {
    assertParameterAllowedStereotype(FLOW);
  }

  @Test
  public void testParameterWithObjectStoreReferences() {
    assertParameterAllowedStereotype(OBJECT_STORE);
  }

  private void assertParameterAllowedStereotype(StereotypeModel parent) {
    org.mule.runtime.api.meta.model.parameter.ParameterModel paramWithReferences =
        mock(org.mule.runtime.api.meta.model.parameter.ParameterModel.class);
    setUpParameter(paramWithReferences, PARAMETER_WITH_REFERENCES_NAME);
    when(paramWithReferences.getAllowedStereotypes())
        .thenReturn(singletonList(StereotypeModelBuilder.newStereotype("config", "test")
            .withParent(parent)
            .build()));

    when(simpleRuntimeParameterGroupModel.getParameterModels()).thenReturn(singletonList(paramWithReferences));

    ConfigurationModel configModel = mock(ConfigurationModel.class);
    when(configModel.getName()).thenReturn(CONFIGURATION_NAME);
    when(configModel.getDescription()).thenReturn(CONFIGURATION_DESCRIPTION);
    when(configModel.getDisplayModel()).thenReturn(of(displayModel));
    when(configModel.getParameterGroupModels()).thenReturn(singletonList(simpleRuntimeParameterGroupModel));

    when(runtimeExtensionModel.getConfigurationModels()).thenReturn(newArrayList(configModel));

    ExtensionModel dto = new ExtensionModelFactory().createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    List<ParameterModel> params = dto.getConfigurationModels().get(0).getParameterGroupModels().get(0).getParameterModels();
    assertThat(params, hasSize(1));

    org.mule.tooling.client.api.extension.model.StereotypeModel ref = params.get(0).getAllowedStereotypes().get(0);
    assertThat(ref.getParent().get().getType(), is(parent.getType()));
    assertThat(ref.getNamespace(), is("TEST"));
    assertThat(ref.getType(), is("CONFIG"));

    boolean isConfig = parent == CONFIG;
    boolean isFlow = parent == FLOW;
    boolean isObjectStore = parent == OBJECT_STORE;

    assertThat(ref.isModuleConfig(), is(isConfig));
    assertThat(ref.isFlow(), is(isFlow));
    assertThat(ref.isObjectStore(), is(isObjectStore));
  }

  @Test
  public void testErrorModel() {
    org.mule.runtime.api.meta.model.error.ErrorModel parentErrorModel =
        new ImmutableErrorModel(PARENT_ERROR_MODEL_TYPE, PARENT_ERROR_MODEL_NAMESPACE, null);
    org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, parentErrorModel);

    when(runtimeExtensionModel.getErrorModels()).thenReturn(singleton(errorModel));

    ExtensionModel dto = new ExtensionModelFactory().createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    Set<ErrorModel> errorModelsDTO = dto.getErrorModels();
    assertThat(errorModelsDTO, hasSize(1));

    ErrorModel errorModelDTO = errorModelsDTO.iterator().next();

    assertThat(errorModelDTO.getType(), is(ERROR_MODEL_TYPE));
    assertThat(errorModelDTO.getNamespace(), is(ERROR_MODEL_NAMESPACE));
    assertThat(errorModelDTO.getParent(), is(not(empty())));
    assertThat(errorModelDTO.getParent().get().getType(), is(PARENT_ERROR_MODEL_TYPE));
    assertThat(errorModelDTO.getParent().get().getNamespace(), is(PARENT_ERROR_MODEL_NAMESPACE));
    assertThat(errorModelDTO.getParent().get().getParent(), is(empty()));
  }

  @Test
  public void testConstructModels() {
    final StereotypeModel stereotypeModel = StereotypeModelBuilder.newStereotype("config", "test").build();

    org.mule.runtime.api.meta.model.error.ErrorModel errorModel =
        new ImmutableErrorModel(ERROR_MODEL_TYPE, ERROR_MODEL_NAMESPACE, null);

    ImmutableNestedComponentModel nestedComponentModel = new ImmutableNestedComponentModel(
                                                                                           "name", "description", null, true,
                                                                                           newHashSet(stereotypeModel),
                                                                                           emptySet());
    ImmutableNestedChainModel nestedChainModel = new ImmutableNestedChainModel(
                                                                               "name", "description", null, true,
                                                                               newHashSet(stereotypeModel), emptySet());
    ImmutableNestedRouteModel nestedRouteModel = new ImmutableNestedRouteModel(
                                                                               "name", "description", emptyList(), null,
                                                                               Int.MinValue(),
                                                                               Int.MaxValue(), ImmutableList
                                                                                   .of(nestedComponentModel, nestedChainModel),
                                                                               emptySet());

    ImmutableConstructModel constructModel = new ImmutableConstructModel(CONSTRUCT_NAME,
                                                                         CONSTRUCT_DESCRIPTION,
                                                                         emptyList(),
                                                                         ImmutableList.of(nestedComponentModel, nestedChainModel,
                                                                                          nestedRouteModel),
                                                                         true,
                                                                         null,
                                                                         newHashSet(errorModel),
                                                                         stereotypeModel,
                                                                         emptySet());

    when(runtimeExtensionModel.getConstructModels()).thenReturn(singletonList(constructModel));

    ExtensionModel dto = new ExtensionModelFactory().createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);

    List<ConstructModel> constructModelsDTO = dto.getConstructModels();
    assertThat(constructModelsDTO, hasSize(1));

    ConstructModel constructModelDTO = constructModelsDTO.get(0);

    assertThat(constructModelDTO.getName(), is(CONSTRUCT_NAME));
    assertThat(constructModelDTO.getDescription(), is(CONSTRUCT_DESCRIPTION));
    assertThat(constructModelDTO.getDisplayModel(), is(empty()));
    assertThat(constructModelDTO.getNestedComponents(), IsCollectionWithSize.hasSize(3));
    assertThat(constructModelDTO.getStereotype().getType(), is(stereotypeModel.getType()));
    assertThat(constructModelDTO.getStereotype().getNamespace(), is(stereotypeModel.getNamespace()));
    assertThat(constructModelDTO.getStereotype().getParent(), is(empty()));

    assertThat(constructModelDTO.getErrorModels(), hasSize(1));
    ErrorModel errorModelDTO = constructModelDTO.getErrorModels().iterator().next();

    assertThat(errorModelDTO.getType(), is(ERROR_MODEL_TYPE));
    assertThat(errorModelDTO.getNamespace(), is(ERROR_MODEL_NAMESPACE));
    assertThat(errorModelDTO.getParent(), is(empty()));

    assertThat(constructModelDTO.getNestedComponents(), IsCollectionWithSize.hasSize(3));

    NestedComponentModel nestedComponentModelDTO = (NestedComponentModel) constructModelDTO.getNestedComponents().get(0);
    assertThat(nestedComponentModelDTO.getName(), is(nestedComponentModel.getName()));
    assertThat(nestedComponentModelDTO.getDescription(), is(nestedComponentModel.getDescription()));
    assertThat(nestedComponentModelDTO.getDisplayModel(), is(nullValue()));
    assertThat(nestedComponentModelDTO.isRequired(), is(true));
    assertThat(nestedComponentModelDTO.getAllowedStereotypes(), IsCollectionWithSize.hasSize(1));

    NestedChainModel nestedChainModelDTO = (NestedChainModel) constructModelDTO.getNestedComponents().get(1);
    assertThat(nestedChainModelDTO.getName(), is(nestedComponentModel.getName()));
    assertThat(nestedChainModelDTO.getDescription(), is(nestedComponentModel.getDescription()));
    assertThat(nestedChainModelDTO.getDisplayModel(), is(nullValue()));
    assertThat(nestedChainModelDTO.isRequired(), is(true));
    assertThat(nestedChainModelDTO.getAllowedStereotypes(), IsCollectionWithSize.hasSize(1));

    NestedRouteModel nestedRouteModelDTO = (NestedRouteModel) constructModelDTO.getNestedComponents().get(2);
    assertThat(nestedRouteModelDTO.getName(), is(nestedComponentModel.getName()));
    assertThat(nestedRouteModelDTO.getDescription(), is(nestedComponentModel.getDescription()));
    assertThat(nestedRouteModelDTO.getDisplayModel(), is(nullValue()));
    assertThat(nestedRouteModelDTO.isRequired(), is(false));
    assertThat(nestedRouteModelDTO.getMinOccurs(), is(nestedRouteModel.getMinOccurs()));
    assertThat(nestedRouteModelDTO.getMaxOccurs(), is(nestedRouteModel.getMaxOccurs()));
    assertThat(nestedRouteModelDTO.getChildComponents(), IsCollectionWithSize.hasSize(2));

    assertThat(nestedRouteModelDTO.getChildComponents().get(0), is(nestedComponentModelDTO));
    assertThat(nestedRouteModelDTO.getChildComponents().get(1), is(nestedChainModelDTO));
  }

  @Test
  public void testFunctionModel() throws Exception {
    ExtensionModel dto = new ExtensionModelFactory().createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertThat(dto.getFunctionModels(), hasSize(1));
    FunctionModel functionDto = dto.getFunctionModels().get(0);
    assertThat(functionDto.getName(), is(FUNCTION_NAME));
    assertThat(functionDto.getDescription(), is(EXTENSION_DESCRIPTION));
    assertThat(functionDto.getDisplayModel().get().getDisplayName(), is(DISPLAY_NAME));
    assertThat(functionDto.getOutput().getType(), is(stringType));
    assertThat(functionDto.getParameterGroupModels(), hasSize(1));
    ParameterGroupModel parameterGroupModel = functionDto.getParameterGroupModels().get(0);
    assertThat(parameterGroupModel.getName(), is(SIMPLE_PARAMETER_GROUP_NAME));
    assertThat(parameterGroupModel.getParameterModels(), hasSize(1));
    assertThat(parameterGroupModel.getParameterModels().get(0).getName(), is(SIMPLE_PARAMETER_NAME));
  }

  @Test
  public void testConfigModel() {
    ConfigurationModel configModel = mock(ConfigurationModel.class);
    when(configModel.getName()).thenReturn(CONFIGURATION_NAME);
    when(configModel.getDescription()).thenReturn(CONFIGURATION_DESCRIPTION);
    when(configModel.getDisplayModel()).thenReturn(of(displayModel));

    ConnectionProviderModel connectionProvider = mock(ConnectionProviderModel.class);
    when(connectionProvider.getName()).thenReturn(CONNECTION_PROVIDER_NAME);
    when(connectionProvider.getDescription()).thenReturn(CONNECTION_PROVIDER_DESCRIPTION);
    when(connectionProvider.getDisplayModel()).thenReturn(of(displayModel));
    when(connectionProvider.getConnectionManagementType()).thenReturn(ConnectionManagementType.POOLING);

    externalLibraryModel = ExternalLibraryModel.builder()
        .withName(EXTERNAL_LIB_MODEL_NAME)
        .withDescription(EXTERNAL_LIB_MODEL_DESCRIPTION)
        .withRegexpMatcher(EXTERNAL_LIB_MODEL_REGEXP_MATCHER)
        .withType(ExternalLibraryType.JAR)
        .withRequiredClassName(String.class.getName())
        .build();
    when(connectionProvider.getExternalLibraryModels()).thenReturn(newHashSet(externalLibraryModel));

    when(configModel.getConnectionProviders()).thenReturn(newArrayList(connectionProvider));

    when(runtimeExtensionModel.getConfigurationModels()).thenReturn(newArrayList(configModel));
    ExtensionModel dto = new ExtensionModelFactory().createExtensionModel(runtimeExtensionModel, MIN_MULE_VERSION);
    assertExtensionModelBasicAttributes(dto);

    assertThat(dto.getConfigurationModels(), hasSize(1));
    final org.mule.tooling.client.api.extension.model.config.ConfigurationModel configurationModel =
        dto.getConfigurationModels().get(0);
    assertThat(configurationModel.getName(), is(CONFIGURATION_NAME));
    assertThat(configurationModel.getDescription(), is(CONFIGURATION_DESCRIPTION));
    assertDisplayModel(configurationModel.getDisplayModel());

    assertThat(configurationModel.getConnectionProviders(), hasSize(1));
    final org.mule.tooling.client.api.extension.model.connection.ConnectionProviderModel connectionProviderModel =
        configurationModel.getConnectionProviders().get(0);
    assertThat(connectionProviderModel.getName(), is(CONNECTION_PROVIDER_NAME));
    assertThat(connectionProviderModel.getDescription(), is(CONNECTION_PROVIDER_DESCRIPTION));
    assertDisplayModel(connectionProviderModel.getDisplayModel());
    assertThat(connectionProviderModel.getConnectionManagementType().isPooling(), is(true));
    assertThat(connectionProviderModel.getConnectionManagementType().isNone(), is(false));
    assertThat(connectionProviderModel.getConnectionManagementType().isCached(), is(false));
    assertThat(connectionProviderModel.getConnectionManagementType().isUnknown(), is(false));

    assertThat(connectionProviderModel.getExternalLibraryModels(), hasSize(1));
    final org.mule.tooling.client.api.extension.model.ExternalLibraryModel externalLibraryModel =
        connectionProviderModel.getExternalLibraryModels().iterator().next();
    assertThat(externalLibraryModel.getName(), is(EXTERNAL_LIB_MODEL_NAME));
    assertThat(externalLibraryModel.getDescription(), is(EXTERNAL_LIB_MODEL_DESCRIPTION));
    assertThat(externalLibraryModel.getRegexMatcher().get(), is(EXTERNAL_LIB_MODEL_REGEXP_MATCHER));
    assertThat(externalLibraryModel.getRequiredClassName().get(), is(String.class.getName()));
    assertThat(externalLibraryModel.getType().isJar(), is(true));
    assertThat(externalLibraryModel.getType().isDependency(), is(false));
    assertThat(externalLibraryModel.getType().isNativeLibrary(), is(false));
    assertThat(externalLibraryModel.getType().isUnknown(), is(false));
  }

  private void initialiseSimpleParameterGroup() {
    setUpParameter(simpleRuntimeParameterModel, SIMPLE_PARAMETER_NAME);
    when(simpleRuntimeParameterGroupModel.getName()).thenReturn(SIMPLE_PARAMETER_GROUP_NAME);
    when(simpleRuntimeParameterGroupModel.isShowInDsl()).thenReturn(false);
    when(simpleRuntimeParameterGroupModel.getExclusiveParametersModels()).thenReturn(emptyList());
    when(simpleRuntimeParameterGroupModel.getParameterModels()).thenReturn(singletonList(simpleRuntimeParameterModel));
    when(simpleRuntimeParameterGroupModel.getDisplayModel()).thenReturn(empty());
    when(simpleRuntimeParameterGroupModel.getLayoutModel()).thenReturn(empty());
  }

  private void setUpParameter(org.mule.runtime.api.meta.model.parameter.ParameterModel mockedParam, String name) {
    when(mockedParam.getName()).thenReturn(name);
    when(mockedParam.getExpressionSupport()).thenReturn(ExpressionSupport.SUPPORTED);
    when(mockedParam.getModelProperty(any())).thenReturn(empty());
    when(mockedParam.getDslConfiguration()).thenReturn(ParameterDslConfiguration.getDefaultInstance());
    when(mockedParam.getLayoutModel()).thenReturn(empty());
    when(mockedParam.getRole()).thenReturn(ParameterRole.BEHAVIOUR);
    when(mockedParam.getDisplayModel()).thenReturn(empty());
    when(mockedParam.getValueProviderModel()).thenReturn(empty());
    when(mockedParam.getAllowedStereotypes()).thenReturn(emptyList());
  }

  private void initialiseExtensionModel() {
    runtimeExtensionModel =
        mock(org.mule.runtime.api.meta.model.ExtensionModel.class);
    when(runtimeExtensionModel.getName()).thenReturn(EXTENSION_NAME);
    when(runtimeExtensionModel.getDescription()).thenReturn(EXTENSION_DESCRIPTION);
    when(runtimeExtensionModel.getVersion()).thenReturn(EXTENSION_VERSION);
    when(runtimeExtensionModel.getVendor()).thenReturn(EXTENSION_VENDOR);
    when(runtimeExtensionModel.getCategory()).thenReturn(Category.CERTIFIED);

    xmlDslModel = XmlDslModel.builder()
        .setPrefix(PREFIX)
        .setNamespace(NAMESPACE)
        .setSchemaVersion(SCHEMA_VERSION)
        .setSchemaLocation(SCHEMA_LOCATION)
        .build();
    when(runtimeExtensionModel.getXmlDslModel()).thenReturn(xmlDslModel);

    PathModel pathModel = new PathModel(FILE, true, EMBEDDED, FILE_EXTENSIONS.toArray(new String[0]));
    displayModel = DisplayModel.builder()
        .displayName(DISPLAY_NAME)
        .example(DISPLAY_EXAMPLE)
        .summary(DISPLAY_SUMMARY)
        .path(pathModel)
        .build();
    when(runtimeExtensionModel.getDisplayModel()).thenReturn(of(displayModel));
  }

  private void initialiseFunction(org.mule.runtime.api.meta.model.function.FunctionModel runtimeFunctionModel) {

    org.mule.runtime.api.meta.model.OutputModel runtimeOutput = mock(org.mule.runtime.api.meta.model.OutputModel.class);
    when(runtimeOutput.getType()).thenReturn(stringType);
    when(runtimeOutput.getModelProperties()).thenReturn(emptySet());
    when(runtimeOutput.getDescription()).thenReturn(EXTENSION_DESCRIPTION);
    when(runtimeOutput.getModelProperty(any())).thenReturn(empty());

    when(runtimeFunctionModel.getName()).thenReturn(FUNCTION_NAME);
    when(runtimeFunctionModel.getDescription()).thenReturn(EXTENSION_DESCRIPTION);
    when(runtimeFunctionModel.getOutput()).thenReturn(runtimeOutput);
    when(runtimeFunctionModel.getModelProperties()).thenReturn(emptySet());
    when(runtimeFunctionModel.getDisplayModel()).thenReturn(Optional.of(displayModel));
    when(runtimeFunctionModel.getParameterGroupModels()).thenReturn(singletonList(simpleRuntimeParameterGroupModel));
  }

  private void assertExtensionModelBasicAttributes(ExtensionModel dto) {
    assertThat(dto.getName(), is(EXTENSION_NAME));
    assertThat(dto.getDescription(), is(EXTENSION_DESCRIPTION));
    assertThat(dto.getVendor(), is(EXTENSION_VENDOR));
    assertThat(dto.getMinMuleVersion(), is(MIN_MULE_VERSION));
    assertThat(dto.getVersion(), is(EXTENSION_VERSION));

    final org.mule.tooling.client.api.extension.model.Category category = dto.getCategory();
    assertThat(category.isCertified(), is(true));
    assertThat(category.isCommunity(), is(false));
    assertThat(category.isPremium(), is(false));
    assertThat(category.isSelect(), is(false));
    assertThat(category.isUnknown(), is(false));
    assertThat(category.getValue(), is(Category.CERTIFIED.name()));

    assertDisplayModel(dto.getDisplayModel());
    assertXmlDslModel(dto.getXmlDslModel());
  }

  private void assertXmlDslModel(org.mule.tooling.client.api.extension.model.XmlDslModel xmlDslModel) {
    assertThat(xmlDslModel.getPrefix(), is(PREFIX));
    assertThat(xmlDslModel.getNamespace(), is(NAMESPACE));
    assertThat(xmlDslModel.getSchemaLocation(), is(SCHEMA_LOCATION));
    assertThat(xmlDslModel.getSchemaVersion(), is(SCHEMA_VERSION));
  }

  private void assertDisplayModel(Optional<org.mule.tooling.client.api.extension.model.DisplayModel> displayModelOptional) {
    assertThat(displayModelOptional.isPresent(), is(true));
    final org.mule.tooling.client.api.extension.model.DisplayModel displayModel = displayModelOptional.get();
    assertThat(displayModel.getDisplayName(), is(DISPLAY_NAME));
    assertThat(displayModel.getSummary(), is(DISPLAY_SUMMARY));
    assertThat(displayModel.getExample(), is(DISPLAY_EXAMPLE));
    assertThat(displayModel.getPathModel().isPresent(), is(true));
    final org.mule.tooling.client.api.extension.model.PathModel pathModel = displayModel.getPathModel().get();
    assertThat(pathModel.getFileExtensions(), hasItem(FILE_EXTENSIONS.get(0)));
    assertThat(pathModel.getType().isFile(), is(true));
    assertThat(pathModel.getType().isDirectory(), is(false));
    assertThat(pathModel.getType().isAny(), is(false));
    assertThat(pathModel.getType().isUnknown(), is(false));
    assertThat(pathModel.getLocation().isEmbedded(), is(true));
    assertThat(pathModel.getLocation().isAny(), is(false));
    assertThat(pathModel.getLocation().isExternal(), is(false));
  }

}
