/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.util.Collections.emptyList;
import static org.mule.tooling.client.internal.LocationFactory.toLocationDTO;
import static org.mule.tooling.client.internal.MetadataPartsFactory.toMetadataFailuresDTO;
import org.mule.datasense.api.metadataprovider.DataSenseMetadataCacheProvider;
import org.mule.runtime.api.component.location.Location;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.runtime.api.util.Pair;
import org.mule.tooling.client.api.datasense.ImmutableMetadataCacheKeyInfo;
import org.mule.tooling.client.api.datasense.ImmutableMetadataResult;
import org.mule.tooling.client.api.datasense.MetadataCache;
import org.mule.tooling.client.api.metadata.MetadataFailure;
import org.mule.tooling.client.internal.datasense.DataSenseArtifact;
import org.mule.tooling.client.internal.serialization.XStreamFactory;

import com.thoughtworks.xstream.XStream;

import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

public class DataSenseMetadataCacheAdapter implements DataSenseMetadataCacheProvider {

  private DataSenseArtifact dataSenseArtifact;

  private XStream xStream;

  public DataSenseMetadataCacheAdapter(DataSenseArtifact dataSenseArtifact) {
    this.dataSenseArtifact = dataSenseArtifact;

    this.xStream = XStreamFactory.createXStream();
  }

  @Override
  public MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(String componentId, Location location,
                                                                                          Long timestamp,
                                                                                          Callable<MetadataResult<ComponentMetadataDescriptor<OperationModel>>> callable) {
    Pair<MetadataCache, Map<String, String>> input = getDataSenseCache(location);

    String result =
        (String) input.getFirst().getOperationMetadata(new ImmutableMetadataCacheKeyInfo(componentId,
                                                                                         toLocationDTO(location).toString(),
                                                                                         timestamp, input.getSecond()),
                                                       () -> {
                                                         final MetadataResult<ComponentMetadataDescriptor<OperationModel>> runtimeMetadataResult =
                                                             callable.call();
                                                         List<MetadataFailure> failures = emptyList();
                                                         if (!runtimeMetadataResult.isSuccess()) {
                                                           failures = toMetadataFailuresDTO(runtimeMetadataResult.getFailures());
                                                         }
                                                         return new ImmutableMetadataResult(runtimeMetadataResult
                                                             .isSuccess(), xStream.toXML(runtimeMetadataResult), failures);
                                                       });

    return (MetadataResult<ComponentMetadataDescriptor<OperationModel>>) xStream.fromXML(result);

  }

  @Override
  public MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(String componentId, Location location,
                                                                                    Long timestamp,
                                                                                    Callable<MetadataResult<ComponentMetadataDescriptor<SourceModel>>> callable) {
    Pair<MetadataCache, Map<String, String>> input = getDataSenseCache(location);

    String result =
        (String) input.getFirst()
            .getSourceMetadata(new ImmutableMetadataCacheKeyInfo(componentId, toLocationDTO(location).toString(), timestamp,
                                                                 input.getSecond()),
                               () -> {
                                 final MetadataResult<ComponentMetadataDescriptor<SourceModel>> runtimeMetadataResult =
                                     callable.call();
                                 List<MetadataFailure> failures = emptyList();
                                 if (!runtimeMetadataResult.isSuccess()) {
                                   failures = toMetadataFailuresDTO(runtimeMetadataResult.getFailures());
                                 }
                                 return new ImmutableMetadataResult(runtimeMetadataResult.isSuccess(),
                                                                    xStream.toXML(runtimeMetadataResult), failures);
                               });

    return (MetadataResult<ComponentMetadataDescriptor<SourceModel>>) xStream.fromXML(result);
  }

  @Override
  public MetadataResult<MetadataKeysContainer> getMetadataKeys(String componentId, Location location, Long timestamp,
                                                               Callable<MetadataResult<MetadataKeysContainer>> callable) {
    Pair<MetadataCache, Map<String, String>> input = getDataSenseCache(location);
    String result =
        (String) input.getFirst()
            .getMetadataKeys(new ImmutableMetadataCacheKeyInfo(componentId, toLocationDTO(location).toString(), timestamp,
                                                               input.getSecond()),
                             () -> {
                               final MetadataResult<MetadataKeysContainer> runtimeMetadataResult = callable.call();
                               List<MetadataFailure> failures = emptyList();
                               if (!runtimeMetadataResult.isSuccess()) {
                                 failures = toMetadataFailuresDTO(runtimeMetadataResult.getFailures());
                               }
                               return new ImmutableMetadataResult(runtimeMetadataResult.isSuccess(),
                                                                  xStream.toXML(runtimeMetadataResult), failures);
                             });

    return (MetadataResult<MetadataKeysContainer>) xStream.fromXML(result);
  }

  private Pair<MetadataCache, Map<String, String>> getDataSenseCache(Location location) {
    Pair<MetadataCache, Map<String, String>> input;
    if (dataSenseArtifact.getParent().isPresent() && dataSenseArtifact.getParent().get().hasComponentModel(location)) {
      DataSenseArtifact parent = this.dataSenseArtifact.getParent().get();
      input = new Pair<>(parent.getMetadataCache(), parent.getProperties());
    } else {
      input = new Pair<>(dataSenseArtifact.getMetadataCache(), dataSenseArtifact.getProperties());
    }
    return input;
  }

}
