/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal;

import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.util.concurrent.TimeUnit.SECONDS;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.mule.tooling.client.internal.utils.ServiceUtils.executeHandlingException;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Preconditions;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.io.File;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cache for {@link ExtensionModel}s
 */
public class ExtensionModelServiceCache implements Disposable {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE =
      "tooling.client.ExtensionModelServiceCache.maximumSize";
  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS =
      "tooling.client.ExtensionModelServiceCache.expireAfterAccess";
  private static final Logger LOGGER = LoggerFactory.getLogger(ExtensionModelServiceCache.class);
  private static final String DEFAULT_MAX_SIZE = "100";

  private List<ExtensionModel> runtimeExtensionModels;
  private Cache<BundleDescriptor, Optional<LoadedExtensionInformation>> extensionModelsByArtifact = createExtensionModelCache();

  private Cache<BundleDescriptor, Optional<LoadedExtensionInformation>> createExtensionModelCache() {
    CacheBuilder builder = CacheBuilder.newBuilder().maximumSize(cacheMaximumSize());
    cacheExpireAfterAccess().ifPresent(value -> builder.expireAfterAccess(value, SECONDS));
    return builder.build();
  }

  private Integer cacheMaximumSize() {
    Integer cacheSize = valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE, DEFAULT_MAX_SIZE));
    Preconditions.checkArgument(cacheSize > 0,
                                format("Wrong value %d provided in system property %s, the cache cannot be less that zero",
                                       cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE));
    return cacheSize;
  }

  private Optional<Long> cacheExpireAfterAccess() {
    if (System.getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS) == null) {
      return empty();
    }

    Long cacheSize = Long.valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_EXPIRE_AFTER_ACCESS));
    Preconditions.checkArgument(cacheSize > 0,
                                format("Wrong value %d provided in system property %s, cacheExpireAfterAccess cannot be less that zero",
                                       cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_MAXIMUM_SIZE));
    return of(cacheSize);
  }

  public ExtensionModelServiceCache() {
    LOGGER.info("Initialising Extension Model Service cache");
  }

  /**
   * Loads an extension model from a {@link ArtifactDescriptor}.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param bundleDescriptor the plugin {@link BundleDescriptor}
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the {@link LoadedExtensionInformation}, empty if the plugin does not have an extension model
   */
  public Optional<LoadedExtensionInformation> loadExtensionInformation(BundleDescriptor bundleDescriptor,
                                                                       InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> extensionModelsByArtifact
        .get(bundleDescriptor, () -> extensionModelService.loadExtensionData(bundleDescriptor)));
  }

  /**
   * Loads an extension model from a {@link File} pointing to the plugin artifact.
   * <p>
   * It uses a cache, so it will only call the {@code extensionModelService} if there's a cache miss.
   *
   * @param plugin the plugin descriptor
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the {@link LoadedExtensionInformation}, empty if the plugin does not have an extension model
   */
  public Optional<LoadedExtensionInformation> loadExtensionInformation(File plugin,
                                                                       InternalExtensionModelService extensionModelService) {
    return executeHandlingException(() -> {
      final ArtifactPluginDescriptor artifactPluginDescriptor = extensionModelService.readBundleDescriptor(plugin);
      return extensionModelsByArtifact
          .get(artifactPluginDescriptor.getBundleDescriptor(),
               () -> extensionModelService.loadExtensionData(artifactPluginDescriptor.getBundleDescriptor()));
    });
  }

  /**
   * Loads the core extension model.
   *
   * @param extensionModelService the service to use to load the {@link ExtensionModel} in case of a cache miss
   * @return the core extension model
   */
  public List<ExtensionModel> loadRuntimeExtensionModels(InternalExtensionModelService extensionModelService) {
    if (runtimeExtensionModels == null) {
      runtimeExtensionModels = extensionModelService.loadRuntimeExtensionModels();
    }
    return runtimeExtensionModels;
  }

  @Override
  public void dispose() {
    LOGGER.info("Disposing Extension Model Service cache");
    this.extensionModelsByArtifact.invalidateAll();
  }

}
