/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.util.Objects.requireNonNull;
import static java.util.Optional.ofNullable;
import static org.apache.commons.io.FileUtils.toFile;
import org.mule.runtime.config.internal.model.ApplicationModel;

import java.net.URI;
import java.util.Optional;

/**
 * Represents an ApplicationModel hierarchy, it could have a parent {@link ToolingApplicationModel}.
 *
 * @since 4.1
 */
public class ToolingApplicationModel {

  private ToolingApplicationModel parent;
  private ApplicationModel muleApplicationModel;
  private ClassLoader classLoader;

  /**
   * Creates an instance of this application model.
   *
   * @param parent defines the parent {@link ToolingApplicationModel}, can be null.
   * @param muleApplicationModel the Mule {@link ApplicationModel}. Non null.
   * @param classLoader the artifact class loader to resolve resources. Non null.
   */
  public ToolingApplicationModel(ToolingApplicationModel parent, ApplicationModel muleApplicationModel, ClassLoader classLoader) {
    requireNonNull(muleApplicationModel, "muleApplicationModel cannot be null");
    requireNonNull(muleApplicationModel, "muleApplicationModel cannot be null");
    this.parent = parent;
    this.muleApplicationModel = muleApplicationModel;
    this.classLoader = classLoader;
  }

  public Optional<ToolingApplicationModel> getParent() {
    return ofNullable(parent);
  }

  public ApplicationModel getMuleApplicationModel() {
    return muleApplicationModel;
  }

  public Optional<URI> findResource(String resource) {
    return ofNullable(classLoader.getResource(resource)).map(url -> toFile(url).toURI());
  }

}
