/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.util.Objects.requireNonNull;
import static java.util.stream.Collectors.toList;
import static org.mule.tooling.client.internal.application.RemoteArtifactContextFactory.createRemoteArtifactContext;
import static org.slf4j.LoggerFactory.getLogger;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.runtime.deployment.model.api.DeployableArtifactDescriptor;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import com.google.common.base.Preconditions;

import java.net.URL;
import java.util.List;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;

/**
 * Base class for {@link Artifact}.
 * 
 * @since 4.1
 */
public abstract class AbstractArtifact<T extends DeployableArtifactDescriptor> implements Artifact {

  private static final Logger LOGGER = getLogger(AbstractArtifact.class);

  private String id;
  private Map<String, String> properties;
  protected ToolingArtifactContext context;

  private LazyValue<List<ExtensionModel>> extensionModels;
  private LazyValue<ToolingArtifactClassLoader> artifactClassLoader;
  private LazyValue<ToolingApplicationModel> applicationModel;

  protected ArtifactResources artifactResources;
  protected T artifactDescriptor;
  protected String remoteArtifactId;

  protected LazyValue<RuntimeToolingService> runtimeToolingService;
  protected ArtifactDeployer artifactDeployer;

  private AtomicBoolean disposed = new AtomicBoolean(false);

  AbstractArtifact(String id, ArtifactResources artifactResources, T artifactDescriptor, ToolingArtifactContext context,
                   Map<String, String> properties) {
    requireNonNull(id, "id cannot be null");
    requireNonNull(artifactResources, "artifactResources cannot be null");
    requireNonNull(artifactDescriptor, "artifactDescriptor cannot be null");
    requireNonNull(context, "context cannot be null");
    requireNonNull(properties, "properties cannot be null");

    this.id = id;
    this.artifactResources = artifactResources;
    this.artifactDescriptor = artifactDescriptor;
    this.context = context;
    this.properties = properties;

    this.applicationModel = newToolingApplicationModelLazyValue();
    this.artifactClassLoader = newToolingArtifactClassLoaderLazyValue();
    this.extensionModels = newExtensionModelsLazyValue();

    this.runtimeToolingService = new LazyValue(() -> context.getRuntimeToolingService());
    this.artifactDeployer = createRemoteArtifactContext(artifactResources, runtimeToolingService);
  }

  @Override
  public String getId() {
    return id;
  }

  @Override
  public ArtifactType getArtifactType() {
    return artifactResources.getArtifactType();
  }

  @Override
  public List<ExtensionModel> getExtensionModels() {
    checkState();
    return extensionModels.get();
  }

  @Override
  public String getArtifactName() {
    checkState();
    return getArtifactClassLoader().getArtifactId();
  }

  @Override
  public Map<String, String> getProperties() {
    return properties;
  }

  @Override
  public URL getArtifactUrlContent() {
    checkState();
    return artifactResources.getArtifactUrlContent();
  }

  @Override
  public ToolingArtifactClassLoader getArtifactClassLoader() {
    checkState();
    return artifactClassLoader.get();
  }

  @Override
  public synchronized void setContext(ToolingArtifactContext context) {
    checkState();
    if (this.context != null && this.context.getAgentConfiguration().isPresent() && context.getAgentConfiguration().isPresent()) {
      if (!this.context.getAgentConfiguration().get().getToolingApiUrl()
          .equals(context.getAgentConfiguration().get().getToolingApiUrl())) {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Application context has changed, trying to dispose (if present) the remote application");
        }
        disposeRemoteArtifactContext();
      }
    }
    this.context = context;
  }

  @Override
  public synchronized void dispose() {
    checkState();
    if (!disposed.getAndSet(true)) {
      disposeClassLoader();
      disposeResources();
      disposeRemoteArtifactContext();
    }
  }

  protected void disposeRemoteArtifactContext() {
    if (remoteArtifactId != null) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("Disposing Mule Runtime remote artifact for artifactId: {}", remoteArtifactId);
      }
      try {
        artifactDeployer.dispose(remoteArtifactId);
      } catch (Exception e) {
        // Nothing to do here...
      }
      remoteArtifactId = null;
    }
  }

  private void disposeResources() {
    if (this.artifactResources != null) {
      this.artifactResources.dispose();
    }
    this.applicationModel = null;
    this.extensionModels = null;
  }

  private void disposeClassLoader() {
    if (artifactClassLoader != null && artifactClassLoader.isComputed()) {
      try {
        if (LOGGER.isDebugEnabled()) {
          LOGGER.debug("Disposing artifact class loader");
        }
        artifactClassLoader.get().dispose();
      } finally {
        artifactClassLoader = null;
      }
    }
  }

  @Override
  public ToolingApplicationModel getApplicationModel() {
    checkState();
    return applicationModel.get();
  }

  protected abstract LazyValue<ToolingArtifactClassLoader> newToolingArtifactClassLoaderLazyValue();

  protected abstract LazyValue<ToolingApplicationModel> newToolingApplicationModelLazyValue();

  private LazyValue<List<ExtensionModel>> newExtensionModelsLazyValue() {
    return new LazyValue<>(() -> {
      final List<ExtensionModel> extensionModels = getArtifactPluginClassLoaders().stream()
          .map(artifactPluginClassLoader -> artifactPluginClassLoader.getArtifactDescriptor().getBundleDescriptor())
          .map(bundleDescriptor -> context.getMuleRuntimeExtensionModelProvider()
              .getExtensionModel(toArtifactDescriptor(bundleDescriptor)).orElse(null))
          .filter(extensionModel -> extensionModel != null)
          .collect(toList());
      extensionModels.addAll(context.getMuleRuntimeExtensionModelProvider().getRuntimeExtensionModels());
      return extensionModels;
    });
  }

  protected List<ArtifactClassLoader> getArtifactPluginClassLoaders() {
    return artifactClassLoader.get().getArtifactPluginClassLoaders();
  }

  private org.mule.tooling.client.api.descriptors.ArtifactDescriptor toArtifactDescriptor(BundleDescriptor bundleDescriptor) {
    final org.mule.tooling.client.api.descriptors.ArtifactDescriptor.Builder builder =
        org.mule.tooling.client.api.descriptors.ArtifactDescriptor.newBuilder()
            .withGroupId(bundleDescriptor.getGroupId())
            .withArtifactId(bundleDescriptor.getArtifactId())
            .withVersion(bundleDescriptor.getVersion())
            .withExtension(bundleDescriptor.getType());
    bundleDescriptor.getClassifier().ifPresent(classifier -> builder.withClassifier(classifier));
    return builder.build();
  }

  @Override
  public String toString() {
    return this.getClass().getSimpleName() + "{artifactName=" + getArtifactName() + ", artifactResources=" + artifactResources
        + "}";
  }

  protected void checkState() {
    Preconditions.checkState(!disposed.get(), "Application already disposed, cannot be used anymore");
  }

}
