/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.RegionPluginClassLoadersFactory;
import org.mule.runtime.deployment.model.internal.application.MuleApplicationClassLoaderFactory;
import org.mule.runtime.deployment.model.internal.nativelib.ArtifactCopyNativeLibraryFinder;
import org.mule.runtime.deployment.model.internal.nativelib.NativeLibraryFinder;
import org.mule.runtime.deployment.model.internal.nativelib.NativeLibraryFinderFactory;
import org.mule.runtime.deployment.model.internal.plugin.PluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingApplicationClassLoaderBuilder;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.File;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Set;

/**
 * Factory that knows how to create a {@link ArtifactClassLoader} for applications.
 *
 * @since 4.0
 */
public class ApplicationClassLoaderFactory {

  private ArtifactClassLoader containerArtifactClassLoader;
  private RegionPluginClassLoadersFactory regionPluginClassLoadersFactory;
  private PluginDependenciesResolver pluginDependenciesResolver;

  public ApplicationClassLoaderFactory(ArtifactClassLoader containerArtifactClassLoader,
                                       RegionPluginClassLoadersFactory regionPluginClassLoadersFactory,
                                       PluginDependenciesResolver pluginDependenciesResolver) {
    requireNonNull(containerArtifactClassLoader, "containerArtifactClassLoader cannot be null");
    requireNonNull(regionPluginClassLoadersFactory, "regionPluginClassLoadersFactory cannot be null");
    requireNonNull(pluginDependenciesResolver, "pluginDependenciesResolver cannot be null");

    this.containerArtifactClassLoader = containerArtifactClassLoader;
    this.regionPluginClassLoadersFactory = regionPluginClassLoadersFactory;
    this.pluginDependenciesResolver = pluginDependenciesResolver;
  }

  public ToolingArtifactClassLoader createApplicationClassLoader(ApplicationDescriptor applicationDescriptor,
                                                                 File workingDirectory,
                                                                 ToolingArtifactClassLoader parentToolingArtifactClassLoader) {
    try {
      ToolingApplicationClassLoaderBuilder builder =
          new ToolingApplicationClassLoaderBuilder(new MuleApplicationClassLoaderFactory(new ToolingNativeLibraryFinderFactory(workingDirectory)),
                                                   regionPluginClassLoadersFactory);

      Set<ArtifactPluginDescriptor> domainPluginDescriptors = new HashSet<>();
      boolean hasDomainDependency = applicationDescriptor.getDomainDescriptor().isPresent();
      if (!hasDomainDependency) {
        builder.setParentClassLoader(containerArtifactClassLoader);
      } else {
        if (hasDomainDependency && parentToolingArtifactClassLoader == null) {
          throw new IllegalStateException(format("'%s' declares a domain dependency '%s' that should have been already resolved"));
        } else {
          if (!applicationDescriptor.getDomainDescriptor().get()
              .equals(parentToolingArtifactClassLoader.getArtifactDescriptor().getBundleDescriptor())) {
            throw new IllegalStateException(
                                            format("Domain class loader to be set for application doesn't match the descriptor, expected: '%s', but was: '%s'",
                                                   applicationDescriptor.getDomainDescriptor().get(),
                                                   parentToolingArtifactClassLoader.getArtifactDescriptor()
                                                       .getBundleDescriptor()));
          }
        }

        for (ArtifactClassLoader artifactPluginClassLoader : parentToolingArtifactClassLoader.getArtifactPluginClassLoaders()) {
          domainPluginDescriptors.add(artifactPluginClassLoader.getArtifactDescriptor());
        }

        builder.setDomainParentClassLoader(parentToolingArtifactClassLoader.getRegionClassLoader());
      }

      pluginDependenciesResolver.resolve(domainPluginDescriptors, new ArrayList<>(applicationDescriptor.getPlugins())).stream()
          .forEach(builder::addArtifactPluginDescriptors);

      builder.setArtifactDescriptor(applicationDescriptor);
      return builder.build();
    } catch (Exception e) {
      throw new ToolingException("Error while creating application class loader", e);
    }
  }

  private class ToolingNativeLibraryFinderFactory implements NativeLibraryFinderFactory {

    private File workingDirectory;

    public ToolingNativeLibraryFinderFactory(File workingDirectory) {
      this.workingDirectory = workingDirectory;
    }

    @Override
    public NativeLibraryFinder create(String name, URL[] urls) {
      return new ArtifactCopyNativeLibraryFinder(new File(new File(workingDirectory, name), "temp"), urls);
    }
  }
}
