/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static org.mule.tooling.client.internal.utils.IOUtils.readContentFromUrl;
import org.mule.runtime.core.api.config.bootstrap.ArtifactType;
import org.mule.tooling.agent.RuntimeToolingService;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Handles an {@link InputStream} with the content of the zipped application from local file system or a remote server.
 *
 * @since 4.0
 */
public class InputStreamArtifactDeployer extends AbstractArtifactDeployer {

  private static final int CONNECT_TIMEOUT = 5000;
  private static final int READ_TIMEOUT = 5000;

  private Function<DeploymentContext, String> deployFunction;
  private Consumer<DeploymentContext> disposeFunction;

  public InputStreamArtifactDeployer(ArtifactResources artifactResources,
                                     Supplier<RuntimeToolingService> runtimeToolingService) {
    super(artifactResources, runtimeToolingService);
    if (artifactResources.getArtifactType().equals(ArtifactType.DOMAIN)) {
      deployFunction =
          (deploymentContext) -> deploymentContext.getRuntimeToolingService().deployDomain(deploymentContext.getContent());
      disposeFunction =
          (deploymentContext) -> deploymentContext.getRuntimeToolingService().disposeDomain(deploymentContext.getArtifactId());
    } else {
      deployFunction = (deploymentContext) -> {
        if (deploymentContext.getDomainId() != null) {
          return deploymentContext.getRuntimeToolingService().deployApplication(deploymentContext.getContent(),
                                                                                deploymentContext.getDomainId());
        }
        return deploymentContext.getRuntimeToolingService().deployApplication(deploymentContext.getContent());
      };
      disposeFunction = (deploymentContext) -> deploymentContext.getRuntimeToolingService()
          .disposeApplication(deploymentContext.getArtifactId());
    }
  }

  @Override
  protected String doInternalDeploy(ArtifactResources artifactResources, RuntimeToolingService runtimeToolingService,
                                    String domainId) {
    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Deploying artifact from URL: {}", artifactResources);
    }

    // TODO this should be configured in Builder for tooling runtimne client (TLS may be needed too)
    byte[] content = readContentFromUrl(artifactResources.getArtifactUrlContent(), CONNECT_TIMEOUT, READ_TIMEOUT);
    return deployFunction.apply(new DeploymentContext(new ByteArrayInputStream(content), domainId, runtimeToolingService));
  }

  @Override
  protected void doInternalDispose(String id, RuntimeToolingService runtimeToolingService) {
    disposeFunction.accept(new DeploymentContext(id, runtimeToolingService));
  }
}
