/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Objects.requireNonNull;
import static org.apache.commons.lang3.ClassUtils.getShortClassName;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.metadata.MetadataAttributes;
import org.mule.runtime.api.metadata.MetadataKeyProvider;
import org.mule.runtime.api.metadata.MetadataKeysContainer;
import org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor;
import org.mule.runtime.api.metadata.resolving.MetadataResult;
import org.mule.tooling.client.api.artifact.ToolingArtifact;
import org.mule.tooling.client.api.component.location.Location;
import org.mule.tooling.client.api.exception.ServiceUnavailableException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.api.metadata.MetadataKeysRequest;
import org.mule.tooling.client.api.request.AbstractToolingRequest;

/**
 * Provides metadata resolution.
 * <p/>
 * This is an internal service and should not be used by clients due to it may change later.
 *
 * @since 4.0
 */
public interface MetadataProvider {

  /**
   * Returns a {@link MetadataResult} of {@link MetadataKeysContainer} described by {@link
   * MetadataKeyProvider#getMetadataKeys()} of the associated {@link MetadataKeyProvider} component identified by the
   * given {@link MetadataKeysRequest}.
   *
   * @param metadataKeysRequest request to get the {@link MetadataKeysContainer}.
   * @return Successful {@link org.mule.tooling.client.api.metadata.MetadataResult} if the keys are successfully resolved Failure {@link org.mule.tooling.client.api.metadata.MetadataResult} if there is an error while resolving the keys.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   * @throws ToolingException if an unexpected error happens.
   */
  MetadataResult<MetadataKeysContainer> getMetadataKeys(MetadataKeysRequest metadataKeysRequest);

  /**
   * Resolves the Metadata description for the operation message processor component identified by the request.
   *
   * @param metadataRequest request to get the {@link ComponentMetadataDescriptor}.
   * @return A {@link ComponentMetadataDescriptor} containing a {@link SourceModel} with its types resolved and
   *         {@link MetadataAttributes} that provides information about the metadata resolution process.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   * @throws ToolingException if an unexpected error happens.
   */
  MetadataResult<ComponentMetadataDescriptor<SourceModel>> getSourceMetadata(MetadataRequest metadataRequest);

  /**
   * Resolves the Metadata description for the operation message processor component identified by the request.
   *
   * @param metadataRequest request to get the {@link ComponentMetadataDescriptor}.
   * @return A {@link ComponentMetadataDescriptor} containing a {@link OperationModel} with its types resolved and
   *         {@link MetadataAttributes} that provides information about the metadata resolution process.
   * @throws ServiceUnavailableException if a connection exception happens when trying to connect to Mule Agent REST API.
   * @throws ToolingException if an unexpected error happens.
   */
  MetadataResult<ComponentMetadataDescriptor<OperationModel>> getOperationMetadata(MetadataRequest metadataRequest);

  /**
   * Request entity for retrieving {@link org.mule.runtime.api.metadata.descriptor.ComponentMetadataDescriptor} for a
   * {@link ToolingArtifact}.
   *
   * @since 1.0
   */
  final class MetadataRequest extends AbstractToolingRequest {

    private Location location;

    /**
     * Sets the location path of component inside the flow.
     *
     * @param location the location path of the component inside the flow.
     */
    public void setLocation(Location location) {
      requireNonNull(location, "componentPath cannot null");
      this.location = location;
    }

    /**
     * @return the location path of the component inside the flow.
     */
    public Location getLocation() {
      return location;
    }

    @Override
    public String toString() {
      return format("%s{requestTimeout=%s,componentLocation=%s}", getShortClassName(this.getClass()), getRequestTimeout(),
                    getLocation());
    }

  }
}
