/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal.application;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.LazyValue;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.internal.plugin.PluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.tooling.client.api.exception.NoSuchApplicationException;
import org.mule.tooling.client.api.exception.ToolingException;
import org.mule.tooling.client.internal.DefaultApplicationModelFactory;
import org.mule.tooling.client.internal.ToolingArtifactContext;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;

import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * Represents a Mule application and allows to be deployed into Mule Runtime.
 * 
 * @since 4.0
 */
public class DefaultApplication extends AbstractArtifact<ApplicationDescriptor> implements Application {

  private Optional<Domain> domain;
  private boolean shouldDisposeDomain = false;

  public DefaultApplication(String id, ArtifactResources artifactResources, ApplicationDescriptor applicationDescriptor,
                            Domain domain, ToolingArtifactContext context, Map<String, String> properties,
                            boolean shouldDisposeDomain) {
    super(id, artifactResources, applicationDescriptor, context, properties);
    if (shouldDisposeDomain && domain == null) {
      throw new IllegalArgumentException("Application cannot be set to dispose the domain without setting a domain reference");
    }
    this.domain = ofNullable(domain);
    this.shouldDisposeDomain = shouldDisposeDomain;

    validatePluginDependencies(context.getPluginDependenciesResolver());
  }

  private void validatePluginDependencies(PluginDependenciesResolver applicationPluginDescriptorsResolver) {
    try {
      domain.ifPresent(domainArtifact -> applicationPluginDescriptorsResolver
          .resolve(domainArtifact.getDescriptor().getPlugins(), new LinkedList<>(this.getDescriptor().getPlugins())));
    } catch (Exception e) {
      throw new ToolingException("Error while creating application", e);
    }
  }

  @Override
  public boolean shouldDisposeDomain() {
    return shouldDisposeDomain;
  }

  @Override
  public Optional<Domain> getDomain() {
    return domain;
  }

  @Override
  public ApplicationDescriptor getDescriptor() {
    return artifactDescriptor;
  }

  protected List<ArtifactClassLoader> getArtifactPluginClassLoaders() {
    if (!domain.isPresent()) {
      return super.getArtifactPluginClassLoaders();
    }
    return ImmutableList.<ArtifactClassLoader>builder()
        .addAll(domain.get().getArtifactClassLoader().getArtifactPluginClassLoaders())
        .addAll(getArtifactClassLoader().getArtifactPluginClassLoaders())
        .build();
  }

  @Override
  public synchronized <R> R evaluateWithRemoteApplication(ApplicationRemoteFunction<R> function) {
    checkState();
    String domainId = null;

    if (domain.isPresent()) {
      domain.get().getRemoteInvokerLock().lock();
      domainId = domain.get().evaluateWithRemoteDomain(
                                                       (deployedDomainId, runtimeToolingService) -> deployedDomainId);
    }
    try {
      if (remoteArtifactId == null) {
        remoteArtifactId = artifactDeployer.deploy(domainId);
      }
      return function.apply(remoteArtifactId, runtimeToolingService.get());
    } catch (NoSuchApplicationException e) {
      if (domain.isPresent()) {
        domainId = domain.get().evaluateWithRemoteDomain(
                                                         (deployedDomainId, runtimeToolingService) -> deployedDomainId);
      }
      remoteArtifactId = artifactDeployer.deploy(domainId);
      return function.apply(remoteArtifactId, runtimeToolingService.get());
    } finally {
      if (domain.isPresent()) {
        domain.get().getRemoteInvokerLock().unlock();
      }
    }
  }

  protected LazyValue<ToolingArtifactClassLoader> newToolingArtifactClassLoaderLazyValue() {
    return new LazyValue<>(() -> context.getApplicationClassLoaderFactory().createApplicationClassLoader(artifactDescriptor,
                                                                                                         artifactResources
                                                                                                             .getWorkingDirectory(),
                                                                                                         domain
                                                                                                             .map(parent -> parent
                                                                                                                 .getArtifactClassLoader())
                                                                                                             .orElse(null)));
  }

  protected LazyValue<ToolingApplicationModel> newToolingApplicationModelLazyValue() {
    return new LazyValue<>(() -> new ToolingApplicationModel(domain.map(parent -> parent.getApplicationModel()).orElse(
                                                                                                                       null),
                                                             new DefaultApplicationModelFactory()
                                                                 .createApplicationModel(artifactDescriptor,
                                                                                         ImmutableSet.<ExtensionModel>builder()
                                                                                             .addAll(getExtensionModels())
                                                                                             .build(),
                                                                                         getArtifactClassLoader()
                                                                                             .getClassLoader(),
                                                                                         null)
                                                                 .orElseThrow(() -> new ToolingException(format("Couldn't create ApplicationModel from %s",
                                                                                                                this))),
                                                             getArtifactClassLoader().getClassLoader()));
  }

}
