/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.tooling.client.internal.dsl;

import static java.lang.Integer.valueOf;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.createDslResolvingContext;
import static org.mule.tooling.client.internal.utils.ExtensionModelUtils.resolveExtensionModels;
import static org.mule.tooling.client.internal.utils.ServiceUtils.executeHandlingException;
import org.mule.maven.client.api.MavenClient;
import org.mule.runtime.api.dsl.DslResolvingContext;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.util.Preconditions;
import org.mule.runtime.extension.api.dsl.syntax.resolver.DslSyntaxResolver;
import org.mule.tooling.client.api.Disposable;
import org.mule.tooling.client.api.descriptors.ArtifactDescriptor;
import org.mule.tooling.client.internal.MuleRuntimeExtensionModelProvider;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.util.List;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Cache for {@link DslResolvingContext} and {@link DslSyntaxResolver}.
 */
public class DslSyntaxServiceCache implements Disposable {

  private static final String TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE =
      "tooling.client.DslSyntaxServiceCache.cache";
  private static final Logger LOGGER = LoggerFactory.getLogger(DslSyntaxServiceCache.class);

  private final Cache<List<ArtifactDescriptor>, DslResolvingContext> dslContextCache;
  private final Cache<ExtensionModel, DslSyntaxResolver> dslResolverCache;

  public DslSyntaxServiceCache() {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Initialising DSL Syntax cache");
    }

    dslContextCache = CacheBuilder.newBuilder()
        .maximumSize(cacheMaximumSize())
        .expireAfterWrite(10, TimeUnit.MINUTES)
        .build();

    dslResolverCache = CacheBuilder.newBuilder()
        .maximumSize(cacheMaximumSize())
        .expireAfterWrite(10, TimeUnit.MINUTES)
        .weakKeys()
        .build();
  }

  private Integer cacheMaximumSize() {
    Integer cacheSize = valueOf(getProperty(TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE, "100"));
    Preconditions.checkArgument(cacheSize > 0,
                                format("Wrong value %d provided in system property %s, the cache cannot be less that zero",
                                       cacheSize, TOOLING_CLIENT_EXTENSION_MODEL_SERVICE_CACHE_CACHE));
    return cacheSize;
  }

  public DslResolvingContext getContext(List<ArtifactDescriptor> artifactDescriptors, MavenClient mavenClient,
                                        MuleRuntimeExtensionModelProvider muleRuntimeExtensionModelProvider) {
    return executeHandlingException(() -> dslContextCache
        .get(artifactDescriptors, () -> createDslResolvingContext(muleRuntimeExtensionModelProvider.getRuntimeExtensionModels(),
                                                                  resolveExtensionModels(mavenClient,
                                                                                         muleRuntimeExtensionModelProvider,
                                                                                         artifactDescriptors))));
  }

  public DslSyntaxResolver getResolver(ExtensionModel extensionModel, DslResolvingContext context) {
    return executeHandlingException(
                                    () -> dslResolverCache.get(extensionModel,
                                                               () -> DslSyntaxResolver.getDefault(extensionModel, context)));
  }

  public void invalidate(List<ArtifactDescriptor> plugins) {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Invalidating DSL Context cache");
    }
    try {
      final DslResolvingContext dslResolvingContext = dslContextCache.getIfPresent(plugins);
      if (dslResolvingContext != null) {
        dslResolvingContext.getExtensions().forEach(dslResolverCache::invalidate);
      }
      dslContextCache.invalidate(plugins);
    } catch (Exception e) {
      LOGGER.error("Failed to invalidate the DSL Context cache: " + e.getMessage(), e);
      dispose();
    }
  }

  public void invalidate(List<ArtifactDescriptor> plugins,
                         org.mule.tooling.client.api.extension.model.ExtensionModel toolingModel) {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Invalidating DSL Resolvers cache for extension: " + toolingModel.getName());
    }

    try {
      dslContextCache.getIfPresent(plugins)
          .getExtension(toolingModel.getName())
          .ifPresent(dslResolverCache::invalidate);
    } catch (Exception e) {
      LOGGER.error("Failed to invalidate the DSL Resolvers cache: " + e.getMessage(), e);
      dispose();
    }
  }

  @Override
  public void dispose() {
    if (LOGGER.isInfoEnabled()) {
      LOGGER.info("Invalidating DSL Syntax cache");
    }
    try {
      this.dslContextCache.invalidateAll();
      this.dslResolverCache.invalidateAll();
    } catch (Exception e) {
      LOGGER.error("Failed to invalidate the DSL Syntax cache: " + e.getMessage(), e);
    }
  }

}
