/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static com.github.tomakehurst.wiremock.client.WireMock.aResponse;
import static com.github.tomakehurst.wiremock.client.WireMock.get;
import static com.github.tomakehurst.wiremock.client.WireMock.urlEqualTo;
import static com.github.tomakehurst.wiremock.core.WireMockConfiguration.wireMockConfig;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static org.apache.commons.io.FileUtils.readFileToString;
import static org.apache.commons.io.FileUtils.toFile;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.sameInstance;
import static org.junit.Assert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyString;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.mule.tooling.client.test.utils.ZipUtils.compress;
import org.mule.runtime.api.deployment.meta.MuleApplicationModel;
import org.mule.runtime.api.deployment.persistence.MuleApplicationModelJsonSerializer;
import org.mule.runtime.deployment.model.api.application.ApplicationDescriptor;
import org.mule.runtime.deployment.model.api.domain.DomainDescriptor;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPluginDescriptor;
import org.mule.runtime.deployment.model.internal.plugin.PluginDependenciesResolver;
import org.mule.runtime.deployment.model.internal.tooling.ToolingArtifactClassLoader;
import org.mule.tooling.agent.RuntimeToolingService;
import org.mule.tooling.client.internal.application.ArtifactResources;
import org.mule.tooling.client.internal.application.DefaultApplication;
import org.mule.tooling.client.internal.application.DefaultDomain;
import org.mule.tooling.client.internal.application.DomainClassLoaderFactory;

import com.github.tomakehurst.wiremock.junit.WireMockClassRule;
import com.google.common.collect.ImmutableList;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;
import java.util.Collection;
import java.util.List;

import io.qameta.allure.Description;
import io.qameta.allure.Feature;
import io.qameta.allure.Story;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;
import org.mockito.stubbing.Answer;

@Feature("Domain")
@Story("DefaultDomain provides the list of extension models, class loader, application model and remote application context")
@RunWith(MockitoJUnitRunner.class)
public class DefaultDomainTestCase {

  private static final String TOOLING_DOMAIN_ID = "toolingDomainId";
  private static final String DEPLOYED_DOMAIN_ID = "domainId";
  private static final String DOMAINS_DATASENSE = "domains/datasense-domain";

  private static final String TOOLING_APP_ID = "toolingAppId";
  private static final String DEPLOYED_APP_ID = "appId";
  private static final String APPLICATIONS_DATASENSE_STATIC = "applications/datasense-static";

  @Mock
  private RuntimeToolingService runtimeToolingService;
  @Mock
  private ToolingArtifactContext context;
  @Mock
  private ToolingArtifactClassLoader domainClassLoader;
  @Mock
  private DomainDescriptor domainDescriptor;
  @Mock
  private DomainClassLoaderFactory domainClassLoaderFactory;
  @Mock
  private PluginDependenciesResolver pluginDependenciesResolver;
  @Rule
  public WireMockClassRule wireMockRule = new WireMockClassRule(wireMockConfig().dynamicPort());
  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();
  @Rule
  public ExpectedException expectedException = none();

  private File domainArtifactFile;
  private File applicationArtifactFile;

  private final String REQUEST_PATH = "/app";

  @Before
  public void before() throws Exception {
    applicationArtifactFile = toFile(this.getClass().getClassLoader().getResource(APPLICATIONS_DATASENSE_STATIC));
    domainArtifactFile = toFile(this.getClass().getClassLoader().getResource(DOMAINS_DATASENSE));

    when(context.getRuntimeToolingService()).thenReturn(runtimeToolingService);
    when(pluginDependenciesResolver.resolve(any(), any())).thenAnswer(
                                                                      (Answer<List<ArtifactPluginDescriptor>>) invocationOnMock -> ImmutableList
                                                                          .<List<ArtifactPluginDescriptor>>builder()
                                                                          .addAll((Collection) invocationOnMock
                                                                              .getArguments()[0])
                                                                          .addAll((Collection) invocationOnMock
                                                                              .getArguments()[1])
                                                                          .build());
  }

  private DomainDescriptor createDomainDescriptor(File applicationLocation) throws IOException {
    final MuleApplicationModel muleApplicationModel = new MuleApplicationModelJsonSerializer().deserialize(
                                                                                                           readFileToString(new File(new File(new File(applicationLocation,
                                                                                                                                                       "META-INF"),
                                                                                                                                              "mule-artifact"),
                                                                                                                                     "mule-artifact.json")));
    DomainDescriptor domainDescriptor = mock(DomainDescriptor.class);
    when(domainDescriptor.getConfigResources()).thenReturn(muleApplicationModel.getConfigs());
    when(domainDescriptor.getPlugins()).thenReturn(emptySet());
    return domainDescriptor;
  }

  private ApplicationDescriptor createApplicationDescriptor(File applicationLocation) throws IOException {
    final MuleApplicationModel muleApplicationModel = new MuleApplicationModelJsonSerializer().deserialize(
                                                                                                           readFileToString(new File(new File(new File(applicationLocation,
                                                                                                                                                       "META-INF"),
                                                                                                                                              "mule-artifact"),
                                                                                                                                     "mule-artifact.json")));
    ApplicationDescriptor applicationDescriptor = mock(ApplicationDescriptor.class);
    when(applicationDescriptor.getConfigResources()).thenReturn(muleApplicationModel.getConfigs());
    return applicationDescriptor;
  }

  @Test
  @Description("Validates that when domain is created from a remote content it will call the deployment with the inputStream")
  public void deploymentRemoteDomainUrlContentShouldBeByInputStream() throws Exception {
    File jarFile = temporaryFolder.newFile();
    compress(jarFile, domainArtifactFile);

    wireMockRule.stubFor(
                         get(urlEqualTo("/app"))
                             .willReturn(aResponse()
                                 .withStatus(200)
                                 .withHeader("Content-Type", "application/octet-stream")
                                 .withBody(IOUtils.toByteArray(jarFile.toURI().toURL()))));
    int port = wireMockRule.port();

    URL applicationContentUrl = new URI("http://localhost:" + port + REQUEST_PATH).toURL();
    when(runtimeToolingService.deployDomain(any(InputStream.class))).thenReturn(DEPLOYED_DOMAIN_ID);

    DefaultDomain domain = new DefaultDomain(TOOLING_DOMAIN_ID, new ArtifactResources(applicationContentUrl),
                                             createDomainDescriptor(domainArtifactFile), context, emptyMap());
    String domainId =
        domain.evaluateWithRemoteDomain((deployedDomainId, RuntimeToolingService) -> deployedDomainId);
    assertThat(domainId, equalTo(DEPLOYED_DOMAIN_ID));

    verify(runtimeToolingService).deployDomain(any(InputStream.class));
  }

  @Test
  @Description("Validates that when domain is created from a local content it will call the deployment with the path")
  public void deploymentLocalDomainUrlContentShouldBeByPath() throws Exception {
    when(runtimeToolingService.deployDomain(any(File.class))).thenReturn(DEPLOYED_DOMAIN_ID);

    DefaultDomain domain = new DefaultDomain(TOOLING_DOMAIN_ID, new ArtifactResources(domainArtifactFile.toURI().toURL()),
                                             createDomainDescriptor(domainArtifactFile), context, emptyMap());
    String domainId =
        domain.evaluateWithRemoteDomain((deployedDomainId, RuntimeToolingService) -> deployedDomainId);
    assertThat(domainId, equalTo(DEPLOYED_DOMAIN_ID));

    verify(runtimeToolingService).deployDomain(any(File.class));
  }

  @Test
  @Description("Validates that when application is created to an existing domain the dispose for application doesn't dispose the domain")
  public void doNotDisposeDomainWhenDisposingApplication() throws IOException {
    when(runtimeToolingService.deployDomain(any(File.class))).thenReturn(DEPLOYED_DOMAIN_ID);
    when(runtimeToolingService.deployApplication(any(File.class), eq(DEPLOYED_DOMAIN_ID))).thenReturn(DEPLOYED_APP_ID);
    when(context.getPluginDependenciesResolver()).thenReturn(pluginDependenciesResolver);

    DefaultDomain domain = new DefaultDomain(TOOLING_DOMAIN_ID, new ArtifactResources(domainArtifactFile.toURI().toURL()),
                                             createDomainDescriptor(domainArtifactFile), context, emptyMap());
    String domainId =
        domain.evaluateWithRemoteDomain((deployedDomainId, RuntimeToolingService) -> deployedDomainId);
    assertThat(domainId, equalTo(DEPLOYED_DOMAIN_ID));

    DefaultApplication application =
        new DefaultApplication(TOOLING_APP_ID, new ArtifactResources(applicationArtifactFile.toURI().toURL()),
                               createApplicationDescriptor(applicationArtifactFile), domain, context, emptyMap(), false);
    String applicationId =
        application.evaluateWithRemoteApplication((deployedApplicationId, RuntimeToolingService) -> deployedApplicationId);
    assertThat(applicationId, equalTo(DEPLOYED_APP_ID));

    application.dispose();

    verify(runtimeToolingService).deployDomain(any(File.class));
    verify(runtimeToolingService).deployApplication(any(File.class), eq(DEPLOYED_DOMAIN_ID));
    verify(runtimeToolingService).disposeApplication(DEPLOYED_APP_ID);
    verify(runtimeToolingService, times(0)).disposeDomain(anyString());
  }

  @Test
  @Description("Validates that if a remote domain was deployed it will be disposed when application is disposed")
  public void disposeRemoteDomainIfDeployed() throws Exception {
    when(runtimeToolingService.deployDomain(any(File.class))).thenReturn(DEPLOYED_DOMAIN_ID);

    DefaultDomain domain = new DefaultDomain(TOOLING_DOMAIN_ID, new ArtifactResources(domainArtifactFile.toURI().toURL()),
                                             createDomainDescriptor(domainArtifactFile), context, emptyMap());
    String domainId =
        domain.evaluateWithRemoteDomain((deployedDomainId, RuntimeToolingService) -> deployedDomainId);
    assertThat(domainId, equalTo(DEPLOYED_DOMAIN_ID));

    domain.dispose();

    verify(runtimeToolingService).deployDomain(any(File.class));
    verify(runtimeToolingService).disposeDomain(DEPLOYED_DOMAIN_ID);
  }

  @Test
  @Description("Validates that class loader for domain should be closed once the domain is disposed")
  public void disposeInvalidatesClassLoader() throws Exception {
    when(domainDescriptor.getRootFolder()).thenReturn(domainArtifactFile);
    when(domainClassLoader.getArtifactDescriptor()).thenReturn(domainDescriptor);
    when(context.getDomainClassLoaderFactory()).thenReturn(domainClassLoaderFactory);
    when(domainClassLoaderFactory.createDomainClassLoader(any(DomainDescriptor.class))).thenReturn(domainClassLoader);

    DefaultDomain domain = new DefaultDomain(TOOLING_DOMAIN_ID, new ArtifactResources(domainArtifactFile.toURI().toURL()),
                                             createDomainDescriptor(domainArtifactFile), context, emptyMap());

    assertThat(domain.getArtifactClassLoader(), sameInstance(domainClassLoader));
    domain.dispose();

    verify(domainClassLoaderFactory).createDomainClassLoader(any(DomainDescriptor.class));
    verify(domainClassLoader).dispose();
  }

}
