/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */

package org.mule.tooling.client.internal;

import static java.lang.String.format;
import static java.util.Collections.emptyMap;
import static java.util.Optional.of;
import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.config.api.XmlConfigurationDocumentLoader.noValidationDocumentLoader;
import static org.mule.runtime.core.api.util.ClassUtils.withContextClassLoader;

import static org.mule.runtime.core.internal.config.RuntimeComponentBuildingDefinitionsUtil.getRuntimeComponentBuildingDefinitionProvider;
import org.mule.runtime.api.component.ConfigurationProperties;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.config.api.XmlConfigurationDocumentLoader;
import org.mule.runtime.config.api.dsl.model.ComponentBuildingDefinitionRegistry;
import org.mule.runtime.config.api.dsl.processor.ArtifactConfig;
import org.mule.runtime.config.api.dsl.processor.ConfigLine;
import org.mule.runtime.config.api.dsl.processor.xml.XmlApplicationParser;
import org.mule.runtime.config.internal.ArtifactConfigResolverContext;
import org.mule.runtime.config.internal.DefaultArtifactConfigResolver;
import org.mule.runtime.config.internal.dsl.model.ClassLoaderResourceProvider;
import org.mule.runtime.config.internal.model.ApplicationModel;
import org.mule.runtime.core.api.config.ConfigResource;
import org.mule.runtime.deployment.model.api.DeployableArtifactDescriptor;
import org.mule.runtime.deployment.model.api.DeploymentException;
import org.mule.runtime.module.artifact.api.classloader.RegionClassLoader;
import org.mule.tooling.client.api.exception.ToolingException;

import java.io.IOException;
import java.net.URL;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * Default implementation for {@link ApplicationModelFactory}.
 *
 * @since 4.0
 */
public class DefaultApplicationModelFactory implements ApplicationModelFactory {

  /**
   /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ApplicationModel> createApplicationModel(DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                           Set<ExtensionModel> extensionModels,
                                                           ClassLoader applicationClassLoader,
                                                           ConfigurationProperties parentConfigurationProperties) {
    try {
      return of(loadApplicationModel(deployableArtifactDescriptor, extensionModels, applicationClassLoader,
                                     parentConfigurationProperties));
    } catch (Exception e) {
      throw new ToolingException("Error while creating application model", e);
    }
  }


  /**
   * Loads {@link ApplicationModel} from {@link ConfigLine}.
   *
   *
   * @param deployableArtifactDescriptor descriptor for the artifact.
   * @param extensionModels {@link Set} of {@link ExtensionModel extension models} to be used for generating the application
   *        model.
   * @param applicationClassLoader {@link ClassLoader} to resolve resources.
   * @param parentConfigurationProperties {@link ConfigurationProperties} for Domain if present.
   * @return a {@link ApplicationModel}.
   */
  private ApplicationModel loadApplicationModel(
                                                DeployableArtifactDescriptor deployableArtifactDescriptor,
                                                Set<ExtensionModel> extensionModels,
                                                ClassLoader applicationClassLoader,
                                                ConfigurationProperties parentConfigurationProperties)
      throws Exception {
    ClassLoader parent = applicationClassLoader.getParent();
    if (!(parent instanceof RegionClassLoader)) {
      throw new DeploymentException(
                                    createStaticMessage(format("The parent of the current artifact classloader must be of type '%s' but found '%s'",
                                                               RegionClassLoader.class.getName(),
                                                               parent.getClass().getName())));
    }

    ArtifactConfig artifactConfig = new DefaultArtifactConfigResolver()
        .resolveArtifactConfig(new ToolingArtifactConfigResolverContext(deployableArtifactDescriptor, extensionModels,
                                                                        (RegionClassLoader) parent, applicationClassLoader));
    Optional<ConfigurationProperties> parentConfigurationPropertiesOptional;
    if (parentConfigurationProperties != null) {
      parentConfigurationPropertiesOptional = of(parentConfigurationProperties);
    } else {
      parentConfigurationPropertiesOptional = of(new EmptyConfigurationProperties());
    }
    return withContextClassLoader(applicationClassLoader, () -> {
      ApplicationModel applicationModel =
          new ApplicationModel(artifactConfig, null, extensionModels, emptyMap(), parentConfigurationPropertiesOptional,
                               of(createComponentBuildingDefinitionRegistry()), false,
                               new ClassLoaderResourceProvider(applicationClassLoader));
      return applicationModel;
    });
  }

  // TODO: EE-5755 - (gfernandes) batch doesn't have an extensionModel therefore use only Mule/MuleEE/Batch component building
  // definition providers to resolve locations and componentTypes
  private static ComponentBuildingDefinitionRegistry createComponentBuildingDefinitionRegistry() {
    final ComponentBuildingDefinitionRegistry componentBuildingDefinitionRegistry =
        new ComponentBuildingDefinitionRegistry();

    getRuntimeComponentBuildingDefinitionProvider()
        .getComponentBuildingDefinitions()
        .forEach(componentBuildingDefinitionRegistry::register);

    return componentBuildingDefinitionRegistry;
  }

  /**
   * Implementation that allows to load an {@link ApplicationModel} without having the properties replaced.
   */
  private class EmptyConfigurationProperties implements ConfigurationProperties {

    @Override
    public <T> Optional<T> resolveProperty(String s) {
      return of((T) "");
    }

    @Override
    public Optional<Boolean> resolveBooleanProperty(String s) {
      return of(false);
    }

    @Override
    public Optional<String> resolveStringProperty(String s) {
      return of("");
    }

  }

  private class ToolingArtifactConfigResolverContext implements ArtifactConfigResolverContext {

    private Set<ExtensionModel> extensionModels;
    private XmlApplicationParser xmlApplicationParser;
    private XmlConfigurationDocumentLoader xmlConfigurationDocumentLoader;
    private DeployableArtifactDescriptor artifactDescriptor;
    private ClassLoader regionClassLoader;
    private ConfigResource[] configResources;

    public ToolingArtifactConfigResolverContext(DeployableArtifactDescriptor artifactDescriptor,
                                                Set<ExtensionModel> extensionModels, RegionClassLoader regionClassLoader,
                                                ClassLoader artifactClassLoader) {
      this.artifactDescriptor = artifactDescriptor;
      this.extensionModels = extensionModels;
      this.regionClassLoader = regionClassLoader;

      this.xmlApplicationParser = XmlApplicationParser.createFromExtensionModels(extensionModels);
      this.xmlConfigurationDocumentLoader = noValidationDocumentLoader();
      this.configResources = artifactDescriptor.getConfigResources().stream()
          .map(configFile -> {
            URL url = artifactClassLoader.getResource(configFile);
            if (url == null) {
              throw new ToolingException(format("Configuration file '%s' cannot be found", configFile));
            }
            try {
              return new ConfigResource(configFile, url.openStream());
            } catch (IOException e) {
              throw new ToolingException(format("Error while reading configuration file '%s'", configFile), e);
            }
          })
          .toArray(ConfigResource[]::new);

    }

    @Override
    public Map<String, String> getArtifactProperties() {
      return emptyMap();
    }

    @Override
    public ConfigResource[] getArtifactConfigResources() {
      return configResources;
    }

    @Override
    public String getArtifactName() {
      return artifactDescriptor.getName();
    }

    @Override
    public XmlConfigurationDocumentLoader getXmlConfigurationDocumentLoader() {
      return xmlConfigurationDocumentLoader;
    }

    @Override
    public Set<ExtensionModel> getExtensions() {
      return extensionModels;
    }

    @Override
    public XmlApplicationParser getXmlApplicationParser() {
      return xmlApplicationParser;
    }

    @Override
    public ClassLoader getExecutionClassLoader() {
      return regionClassLoader;
    }

  }
}
